"""
Pannello delle impostazioni per l'applicazione.
"""
from PyQt5.QtWidgets import QWidget, QVBoxLayout, QLabel, QComboBox, QPushButton, QCheckBox, QSpinBox, QHBoxLayout
from PyQt5.QtCore import Qt
from config.theme import Theme
import json
import os

class SettingsPanel(QWidget):
    """Pannello per la gestione delle impostazioni dell'applicazione."""
    def __init__(self, main_window):
        super().__init__()
        self.main_window = main_window
        self.settings_file = "user_settings.json"
        self.settings = self.load_settings()
        self.init_ui()

    def init_ui(self):
        """Inizializza gli elementi dell'interfaccia utente."""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(20, 20, 20, 20)
        layout.setSpacing(20)

        # Titolo
        title = QLabel("Impostazioni")
        title.setStyleSheet(f"font-size: 18px; font-weight: bold; color: {Theme.TEXT_PRIMARY};")
        layout.addWidget(title)

        # Selezione del player
        player_layout = QHBoxLayout()
        player_label = QLabel("Player predefinito:")
        player_label.setStyleSheet(f"color: {Theme.TEXT_PRIMARY};")
        self.player_combo = QComboBox()
        self.player_combo.addItems(["Browser predefinito", "VLC"])
        
        # Imposta il valore corrente dalle impostazioni salvate
        current_player = self.settings.get("default_player", "Browser predefinito")
        index = self.player_combo.findText(current_player)
        if index >= 0:
            self.player_combo.setCurrentIndex(index)
        
        # Connetti il cambio immediato
        self.player_combo.currentTextChanged.connect(self.on_player_changed)
        
        player_layout.addWidget(player_label)
        player_layout.addWidget(self.player_combo)
        layout.addLayout(player_layout)

        # Pulsante di salvataggio
        self.save_btn = QPushButton("Salva impostazioni")
        self.save_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.PRIMARY};
                color: white;
                border: none;
                border-radius: 4px;
                padding: 8px 16px;
                font-family: 'Inter';
                font-size: 14px;
            }}
            QPushButton:hover {{ background: {Theme.PRIMARY_HOVER}; }}
        """)
        self.save_btn.clicked.connect(self.save_settings)
        layout.addWidget(self.save_btn)
        layout.addStretch()

    def on_player_changed(self, player_name):
        """Gestisce il cambio immediato del player."""
        self.settings["default_player"] = player_name
        self.save_settings_to_file()
        
        # Aggiorna immediatamente il comportamento dell'applicazione
        self.update_player_behavior(player_name)
        
        print(f"Player cambiato in: {player_name}")

    def update_player_behavior(self, player_name):
        """Aggiorna il comportamento dell'applicazione in base al player selezionato."""
        # Qui dovresti aggiornare la logica della tua applicazione principale
        # Ad esempio, puoi aggiornare il main_window o altri componenti
        if hasattr(self.main_window, 'set_default_player'):
            self.main_window.set_default_player(player_name)
        
        # Oppure puoi salvare in una variabile globale accessibile
        # da tutte le parti dell'applicazione
        self.main_window.default_player = player_name

    def load_settings(self):
        """Carica le impostazioni dal file."""
        try:
            if os.path.exists(self.settings_file):
                with open(self.settings_file, 'r', encoding='utf-8') as f:
                    return json.load(f)
        except Exception as e:
            print(f"Errore nel caricamento delle impostazioni: {e}")
        
        # Impostazioni di default
        return {
            "default_player": "Browser predefinito"
        }

    def save_settings_to_file(self):
        """Salva le impostazioni nel file."""
        try:
            with open(self.settings_file, 'w', encoding='utf-8') as f:
                json.dump(self.settings, f, indent=4, ensure_ascii=False)
        except Exception as e:
            print(f"Errore nel salvataggio delle impostazioni: {e}")

    def save_settings(self):
        """Salva le impostazioni correnti."""
        self.settings["default_player"] = self.player_combo.currentText()
        self.save_settings_to_file()
        
        # Aggiorna il comportamento
        self.update_player_behavior(self.settings["default_player"])
        
        print("Impostazioni salvate e applicate")

    def get_default_player(self):
        """Restituisce il player predefinito corrente."""
        return self.settings.get("default_player", "Browser predefinito")

    def open_with_default_player(self, url_or_file):
        """Apre un URL o file con il player predefinito."""
        player = self.get_default_player()
        
        if player == "VLC":
            self.open_with_vlc(url_or_file)
        else:
            self.open_with_browser(url_or_file)

    def open_with_vlc(self, url_or_file):
        """Apre con VLC."""
        try:
            import subprocess
            import platform
            
            system = platform.system()
            if system == "Windows":
                subprocess.Popen(['vlc', url_or_file])
            elif system == "Darwin":  # macOS
                subprocess.Popen(['open', '-a', 'VLC', url_or_file])
            else:  # Linux
                subprocess.Popen(['vlc', url_or_file])
                
            print(f"Apertura con VLC: {url_or_file}")
        except Exception as e:
            print(f"Errore nell'apertura con VLC: {e}")
            # Fallback al browser
            self.open_with_browser(url_or_file)

    def open_with_browser(self, url):
        """Apre con il browser predefinito."""
        try:
            import webbrowser
            webbrowser.open(url)
            print(f"Apertura con browser: {url}")
        except Exception as e:
            print(f"Errore nell'apertura con browser: {e}")
