"""
Search panel with VLC integration, queue support, infinite scroll, and advanced search.
Supports both keyword search and direct PeerTube/PurpleTube video URLs.
Designed to work with a toolbar search bar (no duplicate search bar in the panel).
"""
import re
from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QLabel, QScrollArea, QGridLayout, QMessageBox, QDialog
)
from PyQt5.QtCore import Qt, pyqtSignal
from core.models import Video
from core.services.search_service import SearchService
from core.ui.components.video_card import VideoCard
from core.ui.components.advanced_search_dialog import AdvancedSearchDialog
from config.theme import Theme

class SearchPanel(QWidget):
    """
    Panel for searching and displaying videos with:
    - Infinite scroll (for all searches)
    - VLC integration
    - Queue support
    - Advanced search
    - Trending videos on load
    - Direct video URL support
    """
    video_card_play_requested = pyqtSignal(Video)  # Signal for playing a video
    add_to_queue_requested = pyqtSignal(Video)     # Signal for adding to queue

    def __init__(self, parent=None):
        super().__init__(parent)
        self.search_service = SearchService()
        self.videos = []
        self._display_results = False
        self.current_page = 1
        self.is_loading = False
        self.query = ""
        self.sort_by = "-match"
        self.total_results = 0
        self.total_pages = 1
        self.is_feed_view = False
        self.is_playlist_view = False
        self.setup_ui()
        # Load trending videos by default
        self.load_trending_videos()

    @property
    def display_results(self):
        """Returns whether results are currently displayed."""
        return self._display_results

    def setup_ui(self):
        """Initialize the UI (no search bar here; use the toolbar search bar)."""
        layout = QVBoxLayout(self)
        # Reduce or remove layout margins if you want the content to touch the edges
        layout.setContentsMargins(0, 0, 0, 0)  # Set to 0 for no margins

        # Results label
        self.results_label = QLabel("Search for videos to get started")
        self.results_label.setStyleSheet(f"color: {Theme.TEXT_SECONDARY}; font-size: 14px; margin: 8px;")
        layout.addWidget(self.results_label)

        # Scroll area for video grid
        self.scroll_area = QScrollArea()
        self.scroll_area.setWidgetResizable(True)
        # Update stylesheet to remove padding/margin and position scrollbar closer to the edge
        self.scroll_area.setStyleSheet(f"""
            QScrollArea {{
                background: {Theme.BACKGROUND};
                border: none;
                padding: 0px;
                margin: 0px;
            }}
            QScrollBar:vertical {{
                margin: 0px 0px 0px 0px;  /* Remove scrollbar margins */
                width: 14px;  /* Optional: Adjust scrollbar width */
            }}
        """)
        self.scroll_content = QWidget()
        self.grid_layout = QGridLayout(self.scroll_content)
        self.grid_layout.setSpacing(16)
        self.grid_layout.setContentsMargins(8, 8, 8, 8)  # Adjust grid margins if needed
        self.scroll_area.setWidget(self.scroll_content)
        layout.addWidget(self.scroll_area)

        # Connect scroll event for infinite scroll
        self.scroll_area.verticalScrollBar().valueChanged.connect(self.on_scroll)


    def is_peertube_url(self, query: str) -> bool:
        """Check if the input is a PeerTube/PurpleTube video URL."""
        pattern = r"(https?://[^\s]+/w/|https?://[^\s]+/videos/watch/)[a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}"
        return bool(re.match(pattern, query))

    def extract_video_uuid_from_url(self, url: str) -> str:
        """Extract the video UUID from a PeerTube/PurpleTube URL."""
        match = re.search(r"([a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12})", url)
        return match.group(1) if match else None

    def open_advanced_search(self):
        """Open the advanced search dialog."""
        dialog = AdvancedSearchDialog(self)
        if dialog.exec_() == QDialog.Accepted:
            search_params = dialog.get_search_params()
            self.perform_advanced_search(search_params)

    def perform_search(self, query, sort_by="-match"):
        """
        Perform a basic search using the provided query and sort option.
        If the query is a URL, fetch the video directly.
        """
        if not query:
            return
        self.is_feed_view = False
        self.sort_by = sort_by
        self.current_page = 1
        self.videos = []
        self.clear_grid()

        # Check if the query is a PeerTube/PurpleTube URL
        if self.is_peertube_url(query):
            video_uuid = self.extract_video_uuid_from_url(query)
            if video_uuid:
                self.results_label.setText(f"Loading video {video_uuid}...")
                try:
                    video = self.search_service.get_video_by_uuid(video_uuid)
                    if video:
                        self.videos = [video]
                        self.total_results = 1
                        self.total_pages = 1
                        self.results_label.setText(f"Found video: {video.title}")
                        self.display_videos()
                        self._display_results = True
                    else:
                        self.results_label.setText(f"No video found with UUID: {video_uuid}")
                        self._display_results = False
                except Exception as e:
                    self.results_label.setText(f"Error loading video: {str(e)}")
                    self._display_results = False
                return

        # Otherwise, perform a normal search
        self.search_videos(query, sort_by)

    def perform_advanced_search(self, params):
        """
        Perform a search using advanced filters.
        Args:
            params: Dictionary of search parameters from the advanced search dialog.
        """
        self.is_feed_view = False
        query = params["query"]
        language = params["language"]
        category = params["category"]
        duration = params["duration"]
        sort_by = params["sort_by"]
        # Map sort options to API parameters
        sort_map = {
            "Best Match": "-match",
            "Most Recent": "-publishedAt",
            "Least Recent": "publishedAt"
        }
        sort_param = sort_map.get(sort_by, "-match")
        self.sort_by = sort_param
        # Update UI
        self.results_label.setText(f"Searching for '{query}' with filters...")
        self.query = query
        self.current_page = 1
        self.videos = []
        # Clear previous results
        self.clear_grid()
        # Perform search
        try:
            self.videos, self.total_results = self.search_service.search_videos(
                query=query,
                page=self.current_page,
                count=50,
                sort_by=sort_param,
                language=language,
                category=category,
                duration=duration
            )
            if self.videos:
                self.total_pages = max(1, (self.total_results + 49) // 50)
                self.results_label.setText(
                    f"Found {len(self.videos)} videos for '{query}' (page {self.current_page}/{self.total_pages})"
                )
                self.display_videos()
                self._display_results = True
            else:
                self.results_label.setText(f"No videos found for '{query}'")
                self._display_results = False
        except Exception as e:
            self.results_label.setText(f"Error searching for '{query}': {str(e)}")
            self._display_results = False

    def search_videos(self, query: str, sort_by="-match"):
        """Search for videos and display results."""
        self.is_feed_view = False
        self.query = query
        self.sort_by = sort_by
        self.current_page = 1
        self.videos = []
        self.results_label.setText(f"Searching for '{query}'...")
        # Clear previous results
        self.clear_grid()
        # Perform search with error handling
        try:
            result = self.search_service.search_videos(
                query=query,
                page=self.current_page,
                count=50,
                sort_by=sort_by
            )
            if isinstance(result, tuple) and len(result) == 2:
                self.videos, self.total_results = result
            elif isinstance(result, list):
                self.videos = result
                self.total_results = len(result)
            if self.videos:
                self.total_pages = max(1, (self.total_results + 49) // 50)
                self.results_label.setText(f"Found {len(self.videos)} videos for '{query}' (page {self.current_page}/{self.total_pages})")
                self.display_videos()
                self._display_results = True
            else:
                self.results_label.setText(f"No videos found for '{query}'")
                self._display_results = False
        except Exception as e:
            self.results_label.setText(f"Error searching for '{query}': {str(e)}")
            self._display_results = False

    def load_trending_videos(self):
        """Load trending videos by default when the panel is initialized."""
        self.is_loading = True
        self.is_feed_view = True
        self.current_query = ""
        self.current_page = 1
        self.results_label.setText("🔥 Loading trending videos...")
        try:
            videos, total_results = self.search_service.fetch_trending_videos(
                page=self.current_page,
                count=50
            )
            self.videos = videos
            self.total_results = total_results
            self.total_pages = max(1, (total_results + 49) // 50)
            if self.videos:
                # Add a "Trending" label for clarity
                trending_label = QLabel("🔥 Trending Videos")
                trending_label.setStyleSheet(f"font-size: 18px; font-weight: bold; color: {Theme.TEXT_PRIMARY}; margin: 10px;")
                self.grid_layout.addWidget(trending_label, 0, 0, 1, -1)
                self.display_videos()
                self.results_label.setText(f"📊 Found {len(self.videos)} trending videos (page 1/{self.total_pages})")
                self._display_results = True
            else:
                self.results_label.setText("No trending videos found")
                self._display_results = False
        except Exception as e:
            self.results_label.setText(f"Error loading trending videos: {str(e)}")
            self._display_results = False
        finally:
            self.is_loading = False

    def load_more_videos(self):
        """Load the next page of videos for infinite scroll."""
        if self.is_loading or len(self.videos) >= self.total_results:
            return
        self.is_loading = True
        self.current_page += 1
        try:
            if self.is_feed_view:
                self.results_label.setText(
                    f"Loading more trending videos (page {self.current_page}/{self.total_pages})..."
                )
                new_videos, _ = self.search_service.fetch_trending_videos(
                    page=self.current_page,
                    count=50
                )
            else:
                self.results_label.setText(
                    f"Loading more results for '{self.query}' (page {self.current_page}/{self.total_pages})..."
                )
                new_videos, _ = self.search_service.search_videos(
                    query=self.query,
                    page=self.current_page,
                    count=50,
                    sort_by=self.sort_by
                )
            self.videos.extend(new_videos)
            self.display_videos()
            self.results_label.setText(
                f"Found {len(self.videos)} videos (page {self.current_page}/{self.total_pages})"
            )
        except Exception as e:
            self.results_label.setText(f"Error loading more videos: {str(e)}")
        finally:
            self.is_loading = False

    def on_scroll(self, value):
        """Handle scroll events for infinite scroll."""
        scroll_bar = self.scroll_area.verticalScrollBar()
        if (not self.is_loading and
            scroll_bar.value() >= scroll_bar.maximum() - 100 and
            len(self.videos) < self.total_results):
            self.load_more_videos()

    def display_videos(self):
        """Display videos in a grid layout with 3 columns for better user experience."""
        # Always use 3 columns for a cleaner, more user-friendly layout
        columns = 3
        
        # Clear existing widgets (except the trending label if present)
        for i in reversed(range(self.grid_layout.count())):
            widget = self.grid_layout.itemAt(i).widget()
            if widget and not (self.is_feed_view and i == 0 and isinstance(widget, QLabel)):
                widget.setParent(None)
        
        # Add new widgets
        start_row = 1 if (self.is_feed_view and self.grid_layout.itemAt(0) and isinstance(self.grid_layout.itemAt(0).widget(), QLabel)) else 0
        
        for idx, video in enumerate(self.videos):
            try:
                card = VideoCard(video, main_window=self.parent())
                card.play_requested.connect(self.video_card_play_requested.emit)
                card.add_to_queue_requested.connect(self.add_to_queue_requested.emit)
                row = start_row + idx // columns
                col = idx % columns
                self.grid_layout.addWidget(card, row, col)
            except Exception as e:
                print(f"Error creating VideoCard for video {idx}: {e}")

    def clear_grid(self):
        """Clear all widgets from the grid layout."""
        for i in reversed(range(self.grid_layout.count())):
            widget = self.grid_layout.itemAt(i).widget()
            if widget:
                widget.setParent(None)

    def display_results(self, videos, query, total_results):
        """
        Public method to display results from an external search.
        This is the method called by MainWindow for programmatic searches.
        """
        self.is_feed_view = False
        self.videos = videos
        self.query = query
        self.total_results = total_results
        self.current_page = 1
        self.total_pages = max(1, (total_results + 49) // 50)
        self.results_label.setText(f"Found {len(videos)} videos for '{query}' (page {self.current_page}/{self.total_pages})")
        # Clear previous results
        self.clear_grid()
        # Display new results
        self.display_videos()
        self._display_results = True
