import sys
from pathlib import Path
from PyQt5.QtWidgets import (
    QMainWindow, QVBoxLayout, QHBoxLayout, QWidget, QLineEdit, QPushButton, QLabel,
    QStackedWidget, QScrollArea, QGridLayout, QApplication, QMessageBox, QInputDialog,
    QComboBox, QFrame, QProgressDialog, QShortcut, QToolBar, QStatusBar, QSizePolicy, QDialog, QAction
)
from PyQt5.QtCore import Qt, QTimer, pyqtSignal, QSize, QThreadPool
from PyQt5.QtGui import QFont, QPalette, QColor, QIcon
import logging
import vlc
import webbrowser
from urllib.parse import urlparse

# Aggiungi la root del progetto al path
project_root = Path(__file__).parent.parent.parent
sys.path.insert(0, str(project_root))

from config.theme import Theme
from config.settings import APP_NAME
from core.ui.components.sidebar import Sidebar
from core.ui.components.video_card import VideoCard
from core.ui.panels.search_panel import SearchPanel
from core.ui.panels.playlist_panel import PlaylistPanel
from core.ui.panels.feed_manager_panel import FeedManagerPanel
from core.ui.panels.feed_content_panel import FeedContentPanel
from core.ui.panels.settings_panel import SettingsPanel
from core.ui.components.advanced_search_dialog import AdvancedSearchDialog
from core.services import SearchService, RSSService
from core.player import ModernVLCPlayer as VLCPlayer
from core.player import DirectURLWorker
from utils.helpers import extract_channel_feed_url

logger = logging.getLogger(__name__)

# Debug temporaneo per le impostazioni
from config.settings import PlayerConfig, SettingsManager
logger.info(f"⚙️ [DEBUG] INIT - LEFT_CLICK_ACTION: {PlayerConfig.LEFT_CLICK_ACTION}")

class MainWindow(QMainWindow):
    """Finestra principale dell'applicazione con navigazione a barra laterale e pannelli di contenuto."""

    def __init__(self):
        super().__init__()
        logger.info("🏠 [DEBUG] MainWindow.__init__ started")

        # Inizializza i servizi
        self.rss_service = RSSService()
        self.search_service = SearchService()

        self.setup_window()
        self.setup_ui()
        self.setup_shortcuts()
        self.setup_menu_bar()

        logger.info("🏠 [DEBUG] MainWindow.__init__ completed")

    def setup_window(self):
        """Configura le proprietà della finestra."""
        self.setWindowTitle(f"{APP_NAME}")
        self.setWindowIcon(QIcon("resources/icons/purpletube.svg"))
        self.resize(1200, 800)
        self.setStyleSheet(Theme.global_stylesheet())

        # Nascondi la toolbar predefinita
        for toolbar in self.findChildren(QToolBar):
            toolbar.hide()

        # Configura la barra di stato
        self.statusBar().setStyleSheet(f"""
            QStatusBar {{
                background: {Theme.SURFACE};
                color: {Theme.TEXT_SECONDARY};
                font-family: 'Inter';
                padding: 6px;
            }}
        """)
        self.statusBar().showMessage("Pronto")

    def setup_ui(self):
        """Inizializza l'interfaccia utente completa."""
        logger.info("🏠 [DEBUG] MainWindow.setup_ui started")

        # Widget centrale e layout principale
        central_widget = QWidget()
        self.setCentralWidget(central_widget)
        main_layout = QHBoxLayout(central_widget)
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.setSpacing(0)

        # Barra laterale
        self.sidebar = Sidebar(self)
        self.sidebar.view_changed.connect(self.on_view_changed)
        main_layout.addWidget(self.sidebar)

        # Area contenuto principale
        content_widget = QWidget()
        content_layout = QVBoxLayout(content_widget)
        content_layout.setContentsMargins(0, 0, 0, 0)
        content_layout.setSpacing(0)

        # Barra di ricerca singola
        search_widget = self.create_search_bar()
        content_layout.addWidget(search_widget)

        # Sezione hero
        self.hero_widget = self.create_hero_section()
        content_layout.addWidget(self.hero_widget)

        # Widget impilato per le diverse viste
        self.stacked_widget = QStackedWidget()
        self.stacked_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        self.stacked_widget.setMinimumSize(800, 500)  # Dimensione minima fissa

        # Inizializza i pannelli
        self.search_panel = SearchPanel(self)
        self.playlist_panel = PlaylistPanel(self)
        self.feed_manager_panel = FeedManagerPanel(self, self.rss_service)
        self.feed_content_panel = FeedContentPanel(self, self.rss_service)
        self.settings_panel = SettingsPanel(self)

        # Imposta dimensioni minime/preferite per tutti i pannelli
        for panel in [
            self.search_panel,
            self.playlist_panel,
            self.feed_manager_panel,
            self.feed_content_panel,
            self.settings_panel
        ]:
            panel.setMinimumSize(800, 500)
            panel.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)

        # Aggiungi i pannelli al widget impilato
        self.stacked_widget.addWidget(self.search_panel)          # Indice 0: Ricerca
        self.stacked_widget.addWidget(self.playlist_panel)       # Indice 1: Playlist
        self.stacked_widget.addWidget(self.feed_manager_panel)   # Indice 2: Gestione Feed
        self.stacked_widget.addWidget(self.feed_content_panel)   # Indice 3: Contenuto Feed
        self.stacked_widget.addWidget(self.settings_panel)        # Indice 4: Impostazioni

        # Contenitore per il player VLC
        self.vlc_container = QWidget()
        self.vlc_container.setStyleSheet(f"background: {Theme.BACKGROUND};")
        self.vlc_container.setMinimumSize(800, 500)
        self.vlc_container.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        vlc_layout = QVBoxLayout(self.vlc_container)
        vlc_layout.setContentsMargins(0, 0, 0, 0)
        vlc_layout.setSpacing(0)

        # Inizializza il player VLC
        self.vlc_player = VLCPlayer(self)
        self.vlc_player.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Expanding)
        vlc_layout.addWidget(self.vlc_player)
        self.stacked_widget.addWidget(self.vlc_container)         # Indice 5: Player VLC

        content_layout.addWidget(self.stacked_widget, stretch=1)

        # Collega i segnali da tutti i pannelli
        self._connect_panel_signals()

        main_layout.addWidget(content_widget, stretch=1)

        logger.info("🏠 [DEBUG] MainWindow.setup_ui completed")

    def _connect_panel_signals(self):
        """Collega i segnali da tutti i pannelli agli handler della finestra principale."""
        # Disconnetti prima tutti i segnali per evitare duplicati
        try:
            self.search_panel.video_card_play_requested.disconnect()
            self.playlist_panel.video_card_play_requested.disconnect()
            self.feed_content_panel.video_card_play_requested.disconnect()
            self.feed_manager_panel.video_card_play_requested.disconnect()
        except:
            pass  # Ignora se non erano connessi

        # Riconnetti i segnali
        self.search_panel.video_card_play_requested.connect(self.play_video)
        self.search_panel.add_to_queue_requested.connect(self.add_to_queue)
        self.playlist_panel.video_card_play_requested.connect(self.play_video)
        self.playlist_panel.add_to_queue_requested.connect(self.add_to_queue)
        self.feed_content_panel.video_card_play_requested.connect(self.play_video)
        self.feed_manager_panel.video_card_play_requested.connect(self.play_video)
        self.feed_manager_panel.add_to_queue_requested.connect(self.add_to_queue)

        # Collega i segnali del gestore feed
        try:
            self.feed_manager_panel.feed_selected.disconnect()
            self.feed_manager_panel.feed_refreshed.disconnect()
        except:
            pass
            
        self.feed_manager_panel.feed_selected.connect(self.on_feed_selected)
        self.feed_manager_panel.feed_refreshed.connect(self.feed_manager_panel.refresh_feeds)

        logger.info("🔌 [DEBUG] Tutti i segnali dei pannelli connessi")

    def on_feed_selected(self, feed_id):
        """Gestisci la selezione del feed da FeedManagerPanel."""
        logger.info(f"📡 [DEBUG] Feed selezionato: {feed_id}")
        feeds = self.rss_service.get_feeds()
        selected_feed = next((feed for feed in feeds if feed.id == feed_id), None)
        if selected_feed:
            self.feed_content_panel.load_feed_content(selected_feed)
            self.stacked_widget.setCurrentIndex(3)  # Passa a FeedContentPanel
            self.statusBar().showMessage(f"📡 Visualizzazione feed: {selected_feed.title}")
        else:
            self.statusBar().showMessage("Feed non trovato")

    def on_video_removed(self, video_id):
        """Gestisci la rimozione di un video dalla playlist."""
        logger.info(f"🗑️ [DEBUG] Video rimosso dalla playlist: {video_id}")
        self.statusBar().showMessage(f"Video rimosso dalla playlist: {video_id}")

    def create_search_bar(self):
        """Crea una barra di ricerca singola con tutte le funzionalità originali."""
        logger.info("🏠 [DEBUG] MainWindow.create_search_bar started")

        widget = QWidget()
        widget.setObjectName("searchBarContainer")
        widget.setStyleSheet(f"""
            QWidget#searchBarContainer {{
                background: {Theme.SURFACE};
                border-bottom: 1px solid {Theme.BORDER};
                padding: {Theme.PADDING_SMALL};
            }}
        """)

        layout = QHBoxLayout(widget)
        layout.setContentsMargins(16, 8, 16, 8)
        layout.setSpacing(8)

        # Input di ricerca
        self.search_bar = QLineEdit()
        self.search_bar.setObjectName("searchBar")
        self.search_bar.setPlaceholderText("Cerca video... o incolla un URL video")
        self.search_bar.setFixedHeight(36)
        self.search_bar.setStyleSheet(Theme.search_bar_style())
        self.search_bar.returnPressed.connect(self.search)

        # Menu a tendina per l'ordinamento
        self.sort_dropdown = QComboBox()
        self.sort_dropdown.setFixedHeight(36)
        self.sort_dropdown.addItems(["Miglior corrispondenza", "Più recenti", "Meno recenti"])
        self.sort_dropdown.setStyleSheet(f"""
            QComboBox {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                border-radius: 6px;
                padding: 0 {Theme.PADDING_MEDIUM};
                color: {Theme.TEXT_PRIMARY};
                font-family: {Theme.FONT_FAMILY};
            }}
            QComboBox::drop-down {{
                border: 0px;
            }}
            QComboBox QAbstractItemView {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                color: {Theme.TEXT_PRIMARY};
                font-family: {Theme.FONT_FAMILY};
            }}
        """)

        # Pulsante di ricerca
        search_btn = QPushButton("Cerca")
        search_btn.setFixedHeight(36)
        search_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.PRIMARY};
                color: white;
                border: none;
                border-radius: 6px;
                padding: 0 {Theme.PADDING_MEDIUM};
                font-family: {Theme.FONT_FAMILY};
            }}
            QPushButton:hover {{
                background: {Theme.PRIMARY_HOVER};
            }}
            QPushButton:pressed {{
                background: {Theme.PRIMARY_PRESSED};
            }}
        """)
        search_btn.clicked.connect(self.search)

        # Pulsante ricerca avanzata
        advanced_search_btn = QPushButton("Avanzata")
        advanced_search_btn.setFixedHeight(36)
        advanced_search_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.PRIMARY};
                color: {Theme.TEXT_PRIMARY};
                border: 1px solid {Theme.BORDER};
                border-radius: 6px;
                padding: 0 {Theme.PADDING_MEDIUM};
                font-family: {Theme.FONT_FAMILY};
            }}
            QPushButton:hover {{
                background: {Theme.PRIMARY_HOVER};
            }}
            QPushButton:pressed {{
                background: {Theme.PRIMARY_PRESSED};
            }}
        """)
        advanced_search_btn.clicked.connect(self.open_advanced_search)

        # Pulsante aggiungi feed
        add_feed_btn = QPushButton("Aggiungi Feed")
        add_feed_btn.setFixedHeight(36)
        add_feed_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.PRIMARY};
                color: white;
                border: none;
                border-radius: 6px;
                padding: 0 {Theme.PADDING_MEDIUM};
                font-family: {Theme.FONT_FAMILY};
            }}
            QPushButton:hover {{
                background: {Theme.PRIMARY_HOVER};
            }}
            QPushButton:pressed {{
                background: {Theme.PRIMARY_PRESSED};
            }}
        """)
        add_feed_btn.clicked.connect(self.add_feed_from_url)

        # Aggiungi widget al layout
        layout.addWidget(self.search_bar, stretch=1)
        layout.addWidget(self.sort_dropdown)
        layout.addWidget(search_btn)
        layout.addWidget(advanced_search_btn)
        layout.addWidget(add_feed_btn)

        logger.info("🏠 [DEBUG] MainWindow.create_search_bar completed")
        return widget

    def create_hero_section(self):
        """Crea la sezione hero con la descrizione dell'app."""
        hero_widget = QFrame()
        hero_widget.setObjectName("heroWidget")
        hero_widget.setFixedHeight(160)
        hero_widget.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Fixed)
        hero_widget.setStyleSheet(f"""
            QFrame#heroWidget {{
                background: qlineargradient(x1:0, y1:0, x2:1, y2:0,
                    stop:0 #6A1B9A, stop:0.5 #8E24AA, stop:1 #4A148C);
                margin-top: 0px;
                padding: {Theme.PADDING_LARGE};
            }}
        """)

        hero_layout = QVBoxLayout(hero_widget)
        hero_layout.setContentsMargins(20, 20, 20, 20)
        hero_layout.setSpacing(8)

        title = QLabel(f"Benvenuto in {APP_NAME}")
        title.setStyleSheet("""
            QLabel {
                font: bold 24px 'Inter';
                color: white;
            }
        """)

        description = QLabel(
            "Scopri video popolari e in tendenza da istanze PeerTube decentralizzate. "
            "Cerca, iscriviti e riproduci video senza tracciamento. #stopchatcontrol"
        )
        description.setStyleSheet("""
            QLabel {
                font: 18px 'Inter';
                color: rgba(255, 255, 255, 0.8);
            }
        """)
        description.setWordWrap(True)

        hero_layout.addWidget(title)
        hero_layout.addWidget(description)

        return hero_widget

    def setup_shortcuts(self):
        """Imposta le scorciatoie da tastiera per l'applicazione."""
        # Navigazione
        QShortcut("Ctrl+F", self, self.search_bar.setFocus)
        QShortcut("Ctrl+Shift+F", self, self.open_advanced_search)
        QShortcut("Ctrl+Shift+A", self, self.add_feed_from_url)
        QShortcut("Ctrl+1", self, lambda: self.on_view_changed("search"))
        QShortcut("Ctrl+2", self, lambda: self.on_view_changed("playlists"))
        QShortcut("Ctrl+3", self, lambda: self.on_view_changed("feeds"))
        QShortcut("Ctrl+4", self, lambda: self.on_view_changed("feed_content"))
        QShortcut("Ctrl+5", self, lambda: self.on_view_changed("settings"))
        QShortcut("Ctrl+6", self, lambda: self.on_view_changed("vlc"))
        QShortcut("Ctrl+Q", self, QApplication.quit)
        QShortcut("Esc", self, lambda: self.on_view_changed("search"))

        # Riproduzione
        QShortcut("Space", self, self.vlc_player.toggle_play_pause)
        QShortcut("Ctrl+Right", self, self.vlc_player.seek_forward)
        QShortcut("Ctrl+Left", self, self.vlc_player.seek_backward)
        QShortcut("Ctrl+Up", self, self.vlc_player.increase_volume)
        QShortcut("Ctrl+Down", self, self.vlc_player.decrease_volume)
        QShortcut("Ctrl+Shift+N", self, self.play_next_in_queue)
        QShortcut("Ctrl+Shift+P", self, self.play_previous_in_queue)

        # Coda
        QShortcut("Ctrl+Shift+Q", self, self.vlc_player.clear_queue)
        QShortcut("Ctrl+Shift+R", self, self.vlc_player.repeat_queue)

        # Varie
        QShortcut("Ctrl+I", self, self.show_about)
        QShortcut("Ctrl+S", self, self.show_shortcuts)

    def setup_menu_bar(self):
        """Imposta la barra dei menu dell'applicazione."""
        menu_bar = self.menuBar()

        # Menu File
        file_menu = menu_bar.addMenu("&File")
        exit_action = QAction("Esci", self)
        exit_action.triggered.connect(QApplication.quit)
        file_menu.addAction(exit_action)

        # Menu Visualizza
        view_menu = menu_bar.addMenu("&Visualizza")
        search_action = QAction("Ricerca", self)
        search_action.triggered.connect(lambda: self.on_view_changed("search"))
        playlists_action = QAction("Playlist", self)
        playlists_action.triggered.connect(lambda: self.on_view_changed("playlists"))
        feeds_action = QAction("Feed", self)
        feeds_action.triggered.connect(lambda: self.on_view_changed("feeds"))
        vlc_action = QAction("Player VLC", self)
        vlc_action.triggered.connect(lambda: self.on_view_changed("vlc"))
        view_menu.addActions([search_action, playlists_action, feeds_action, vlc_action])

        # Menu Aiuto
        help_menu = menu_bar.addMenu("&Aiuto")
        about_action = QAction("Informazioni", self)
        about_action.triggered.connect(self.show_about)
        shortcuts_action = QAction("Scorciatoie", self)
        shortcuts_action.triggered.connect(self.show_shortcuts)
        help_menu.addActions([about_action, shortcuts_action])

    def on_view_changed(self, view_name):
        """Gestisci il cambio di vista tra i diversi pannelli."""
        logger.info(f"🔄 [DEBUG] Vista cambiata in: {view_name}")

        view_index = {
            "search": 0,
            "playlists": 1,
            "subscriptions": 2,
            "feeds": 2,
            "feed_content": 3,
            "settings": 4,
            "vlc": 5
        }.get(view_name, 0)

        self.stacked_widget.setCurrentIndex(view_index)

        # Nascondi la sezione hero per VLC, Playlist e Feed
        self.hero_widget.setVisible(view_name not in ["vlc", "playlists", "subscriptions", "feeds"])

        if view_name in ["subscriptions", "feeds"]:
            self.feed_manager_panel.refresh_feeds()
        elif view_name == "playlists":
            self.playlist_panel.refresh_playlists()

        if view_name == "vlc":
            QTimer.singleShot(100, self.resize_vlc_player)

        self.statusBar().showMessage(f"Vista: {view_name.capitalize()}")

    def resize_vlc_player(self):
        """Ridimensiona il player VLC solo se la vista corrente è 'vlc'."""
        if self.stacked_widget.currentIndex() == 5 and hasattr(self, 'vlc_player'):
            logger.info(f"🎬 [DEBUG] Ridimensionamento player VLC a: {self.vlc_container.size()}")
            self.vlc_player.resize(self.vlc_container.size())
            self.vlc_player.reattach_video_surface()

    def resizeEvent(self, event):
        """Gestisci gli eventi di ridimensionamento della finestra."""
        super().resizeEvent(event)
        logger.info(f"📏 [DEBUG] Dimensione finestra: {self.size()}")
        logger.info(f"📏 [DEBUG] Dimensione stacked_widget: {self.stacked_widget.size()}")
        logger.info(f"📏 [DEBUG] Dimensione pannello corrente: {self.stacked_widget.currentWidget().size()}")

        if self.stacked_widget.currentIndex() == 5:
            QTimer.singleShot(100, self.resize_vlc_player)

    def search(self):
        """Esegui una ricerca o riproduci un video se l'input è un URL."""
        logger.info("🔍 [DEBUG] MainWindow.search started")

        query = self.search_bar.text().strip()
        if not query:
            self.statusBar().showMessage("Inserisci un termine di ricerca o incolla un URL video...")
            return

        # Controlla se l'input è un URL
        if self.is_valid_url(query):
            self.play_video(query)
            return

        # Altrimenti, esegui la ricerca
        sort_option = self.sort_dropdown.currentText()
        sort_param = {
            "Miglior corrispondenza": "-match",
            "Più recenti": "-createdAt",
            "Meno recenti": "createdAt"
        }.get(sort_option, "-match")

        self.search_panel.perform_search(query, sort_param)

    def is_valid_url(self, query):
        """Controlla se l'input è un URL valido."""
        try:
            result = urlparse(query)
            return all([result.scheme, result.netloc])
        except:
            return False

    def open_advanced_search(self):
        """Apri la finestra di dialogo per la ricerca avanzata e processa i resultados."""
        logger.info("⚙️ [DEBUG] Apertura finestra di dialogo ricerca avanzata")

        dialog = AdvancedSearchDialog(self)
        if dialog.exec_() == QDialog.Accepted:
            search_params = dialog.get_search_params()
            self.search_panel.perform_advanced_search(search_params)

    def add_feed_from_url(self):
        """Aggiungi un feed da URL."""
        logger.info("📺 [DEBUG] Aggiunta feed da URL")

        url, ok = QInputDialog.getText(
            self,
            "Aggiungi Feed RSS",
            "Incollare l'URL del canale PeerTube o del feed RSS:",
            text="https://example.com/feeds/videos.xml"
        )

        if ok and url:
            try:
                self.statusBar().showMessage(f"🔄 Aggiunta feed: {url}...")

                if "/feeds/videos.xml" in url:
                    rss_url = url
                else:
                    rss_url = extract_channel_feed_url(url)

                if rss_url:
                    if self.rss_service.add_feed(rss_url):
                        QMessageBox.information(self, "Successo", f"Feed aggiunto: {rss_url}")
                        self.feed_manager_panel.refresh_feeds()
                    else:
                        QMessageBox.warning(self, "Attenzione", "Impossibile aggiungere il feed")
                else:
                    QMessageBox.warning(self, "Attenzione", "Impossibile estrarre l'URL del feed")

            except Exception as e:
                QMessageBox.warning(self, "Attenzione", f"Errore: {str(e)[:100]}")
                logger.error(f"❌ [DEBUG] Errore nell'aggiunta del feed: {e}")

            finally:
                self.statusBar().showMessage("Pronto")

    def play_video(self, video):
        """Play video according to user's preference."""
        logger.info(f"🎬 [DEBUG] play_video chiamato con: {getattr(video, 'title', video)}")

        # Get the default player from the settings panel
        default_player = self.settings_panel.get_default_player() if hasattr(self, 'settings_panel') else "Browser predefinito"
        logger.info(f"🎬 [DEBUG] Default player setting: {default_player}")

        if default_player == "VLC":
            logger.info("🎬 [DEBUG] Scelta: Riproduci con VLC")
            self._play_with_vlc_direct(video)
        else:
            logger.info("🎬 [DEBUG] Scelta: Apri nel browser")
            # Open in browser
            if isinstance(video, str):
                url = video
                title = "Video"
            else:
                url = video.url
                title = getattr(video, 'title', 'Sconosciuto')
            logger.info(f"🎬 [DEBUG] Apertura browser con URL: {url}")
            webbrowser.open(url)
            self.statusBar().showMessage(f"🌐 Aperto nel browser: {title}")

    def _play_with_vlc_direct(self, video):
        """Riproduci il video con VLC."""
        logger.info(f"🎬 [DEBUG] _play_with_vlc_direct ENTRATA per: {getattr(video, 'title', 'Sconosciuto')}")
        
        if isinstance(video, str):
            url = video
            title = "Video"
        else:
            url = video.url
            title = getattr(video, 'title', 'Sconosciuto')

        logger.info(f"🎬 [DEBUG] _play_with_vlc_direct - URL: {url}")
        logger.info(f"🎬 [DEBUG] _play_with_vlc_direct - Titolo: {title}")

        def on_direct_url_ready(direct_url: str):
            logger.info(f"🎬 [DEBUG] URL risolto per VLC diretto: {direct_url}")
            QTimer.singleShot(0, lambda: self._safe_play_video(video, direct_url, title))

        def on_direct_url_error(error_msg: str):
            logger.error(f"❌ [DEBUG] Errore nella risoluzione VLC diretto {url}: {error_msg}")
            self.statusBar().showMessage(f"⚠️ Errore nella riproduzione VLC di {title}: {error_msg}")
            
            # Fallback al browser
            QMessageBox.warning(
                self,
                "Errore di riproduzione VLC",
                f"Impossibile riprodurre {title} con VLC ({url}): {error_msg}\n\n"
                "Tentativo di riproduzione nel browser..."
            )
            self._open_with_browser(url)

        logger.info("🎬 [DEBUG] Avvio worker DirectURLWorker")
        self.worker = DirectURLWorker(url)
        self.worker.signals.finished.connect(on_direct_url_ready)
        self.worker.signals.error.connect(on_direct_url_error)
        QThreadPool.globalInstance().start(self.worker)
        logger.info("🎬 [DEBUG] Worker DirectURLWorker avviato")

    def _safe_play_video(self, video, direct_url: str, title: str):
        """Riproduci il video nel player VLC in modo sicuro."""
        logger.info(f"🎬 [DEBUG] _safe_play_video ENTRATA - URL: {direct_url}")
        logger.info(f"🎬 [DEBUG] _safe_play_video - VLC player stato: {hasattr(self, 'vlc_player')}")
        
        self.vlc_player.play(video=video, url=direct_url)
        self.on_view_changed("vlc")
        self.statusBar().showMessage(f"🎬 Ora in riproduzione: {title}")
        logger.info("🎬 [DEBUG] _safe_play_video COMPLETATA")

    def _open_with_browser(self, url):
        """Apri l'URL nel browser predefinito."""
        logger.info(f"🌐 [DEBUG] Apertura nel browser: {url}")
        try:
            webbrowser.open(url)
            self.statusBar().showMessage(f"Aperto nel browser: {url}")
        except Exception as e:
            logger.error(f"❌ [DEBUG] Errore nell'apertura del browser: {e}")
            self.statusBar().showMessage(f"Errore nell'apertura del browser: {str(e)}")

    def add_to_queue(self, video):
        """Aggiungi un video alla coda del player VLC."""
        logger.info(f"🎬 [DEBUG] add_to_queue chiamato con: {getattr(video, 'title', 'Sconosciuto')}")
        self.vlc_player.queue_manager.add_video(video)
        self.statusBar().showMessage(f"Aggiunto alla coda: {getattr(video, 'title', 'Sconosciuto')}")

    def play_next_in_queue(self):
        """Riproduci il prossimo video nella coda."""
        logger.info("🎬 [DEBUG] Riproduzione del prossimo video nella coda")

        if hasattr(self.vlc_player, 'queue_manager') and self.vlc_player.queue_manager:
            if self.vlc_player.queue_manager.play_next():
                self.on_view_changed("vlc")
            else:
                self.statusBar().showMessage("La coda è vuota")
        else:
            self.statusBar().showMessage("Gestore coda non pronto")

    def play_previous_in_queue(self):
        """Riproduci il video precedente nella coda."""
        logger.info("🎬 [DEBUG] Riproduzione del video precedente nella coda")
        self.statusBar().showMessage("Funzionalità video precedente non ancora implementata")

    def show_about(self):
        """Mostra la finestra di dialogo con le informazioni sull'app."""
        about_text = f"""
        <table style="border-spacing: 10px; border-collapse: separate; width: 100%;">
            <tr>
                <td style="vertical-align: top; padding-right: 15px; padding-left: 5px;">
                    <img src="resources/icons/purpletube.svg" width="64" height="64" />
                </td>
                <td style="vertical-align: top;">
                    <h2 style="font-family: 'Inter'; font-size: 16pt; font-weight: bold; color: {Theme.TEXT_PRIMARY}; margin-top: 0; margin-bottom: 5px;">
                        {APP_NAME}
                    </h2>
                    <p style="font-family: 'Inter'; font-size: 11pt; color: {Theme.TEXT_SECONDARY}; margin-top: 0; margin-bottom: 5px;">
                        Un lettore video open-source per contenuti decentralizzati PeerTube, progettato per promuovere la libertà digitale e l'accesso etico ai media.
                    </p>
                    <p style="font-family: 'Inter'; font-size: 11pt; color: {Theme.TEXT_SECONDARY}; margin-top: 0; margin-bottom: 5px;">
                        <strong>Licenza:</strong> <a href="https://www.gnu.org/licenses/gpl-3.0.en.html" style="color: {Theme.PRIMARY}; text-decoration: none;">GNU GPL v3</a>
                    </p>
                    <p style="font-family: 'Inter'; font-size: 11pt; color: {Theme.TEXT_SECONDARY}; margin-top: 0; margin-bottom: 5px;">
                        <strong>Sviluppatore:</strong> Angelo Massaro
                    </p>
                    <p style="font-family: 'Inter'; font-size: 11pt; color: {Theme.TEXT_SECONDARY}; margin-top: 0; margin-bottom: 5px;">
                        Per segnalare bug o richiedere informazioni, contatta: <a href="mailto:amassaro@boostmedia.it" style="color: {Theme.PRIMARY}; text-decoration: none;">amassaro@boostmedia.it</a>
                    </p>
                    <p style="font-family: 'Inter'; font-size: 11pt; color: {Theme.TEXT_SECONDARY}; margin-top: 0; margin-bottom: 5px;">
                        Parte del progetto <a href="https://boostmedia.it/" style="color: {Theme.PRIMARY}; text-decoration: none;">BoostMedia APS</a>, un'iniziativa per promuovere strumenti digitali etici e open-source.
                    </p>
                </td>
            </tr>
        </table>
        """

        msg_box = QMessageBox(self)
        msg_box.setWindowTitle("Informazioni")
        msg_box.setTextFormat(Qt.RichText)
        msg_box.setText(about_text)

        msg_box.setMinimumWidth(600)

        msg_box.exec_()




    def show_shortcuts(self):
        """Mostra una finestra di dialogo con tutte le scorciatoie da tastiera."""
        shortcuts_text = """
        <h2>Scorciatoie da Tastiera</h2>
        <table border="0" cellpadding="5">
            <tr><td><b>Ctrl+F</b></td><td>Focus sulla barra di ricerca</td></tr>
            <tr><td><b>Ctrl+Shift+F</b></td><td>Ricerca avanzata</td></tr>
            <tr><td><b>Ctrl+Shift+A</b></td><td>Aggiungi feed RSS</td></tr>
            <tr><td><b>Ctrl+1-6</b></td><td>Cambia vista (Ricerca, Playlist, Feed, Contenuto Feed, Impostazioni, VLC)</td></tr>
            <tr><td><b>Spazio</b></td><td>Play/Pausa</td></tr>
            <tr><td><b>Ctrl+Freccia Destra/Sinistra</b></td><td>Avanti/Indietro</td></tr>
            <tr><td><b>Ctrl+Freccia Su/Giù</b></td><td>Aumenta/Diminuisci volume</td></tr>
            <tr><td><b>F</b></td><td>Attiva/Disattiva schermo intero</td></tr>
            <tr><td><b>Ctrl+Shift+N/P</b></td><td>Prossimo/Precedente nella coda</td></tr>
            <tr><td><b>Ctrl+Shift+Q</b></td><td>Svuota coda</td></tr>
            <tr><td><b>Ctrl+Shift+R</b></td><td>Ripeti coda</td></tr>
            <tr><td><b>Ctrl+I</b></td><td>Mostra informazioni</td></tr>
            <tr><td><b>Ctrl+S</b></td><td>Mostra scorciatoie</td></tr>
            <tr><td><b>Esc</b></td><td>Esci dallo schermo intero o torna alla ricerca</td></tr>
        </table>
        """
        QMessageBox.about(self, "Scorciatoie da Tastiera", shortcuts_text)
