import requests
from typing import List, Tuple, Optional, Dict, Any
from functools import lru_cache
from urllib.parse import urlencode
from datetime import datetime
from core.models import Video
from config.settings import SEPIASEARCH_BASE_URL, DEFAULT_SEARCH_COUNT
import logging

logger = logging.getLogger(__name__)

class SearchService:
    """Search service for fetching videos from SepiaSearch API."""
    SESSION = requests.Session()
    SESSION.headers.update({"User-Agent": "purpleTubePlayer/1.0"})

    @staticmethod
    @lru_cache(maxsize=32)
    def search_videos(
        query: str,
        page: int = 1,
        count: int = DEFAULT_SEARCH_COUNT,
        sort_by: str = "-match",
        language: Optional[str] = None,
        category: Optional[str] = None,
        duration: Optional[str] = None,
        published_after: Optional[str] = None,
        _cache_key: str = datetime.now().strftime("%Y-%m-%d-%H")  # Cache per hour
    ) -> Tuple[List[Video], int]:
        """
        Search for videos using the SepiaSearch API.
        Args:
            query: Search query string.
            page: Page number for pagination.
            count: Number of results per page.
            sort_by: Sorting criteria.
            language: Language filter.
            category: Category filter.
            duration: Duration filter.
            published_after: Publication date filter.
            _cache_key: Internal use for cache busting.
        Returns:
            Tuple of (List[Video], total_results).
        """
        logger.info(f"Starting search for: '{query}' (page {page}, count {count})")
        try:
            # Build request parameters
            params = {
                'search': query,
                'start': (page - 1) * count,
                'count': count,
                'nsfw': 'false',
                'sort': sort_by
            }
            # Add optional filters
            if language:
                params['languageOneOf'] = language
            if duration:
                params['durationRange'] = duration
            if category:
                params['categoryOneOf'] = category
            if published_after:
                params['publishedAfter'] = published_after
            logger.debug(f"Request params: {params}")
            # Make API request
            url = f"{SEPIASEARCH_BASE_URL}?{urlencode(params)}"
            logger.debug(f"Making request to: {url}")
            resp = SearchService.SESSION.get(url, timeout=10)
            resp.raise_for_status()
            data = resp.json()
            logger.debug(f"Response data keys: {list(data.keys()) if isinstance(data, dict) else 'Not a dict'}")
            # Parse videos from response
            videos = []
            items = data.get("data", [])
            logger.debug(f"Found {len(items)} items in response")
            for idx, item in enumerate(items):
                try:
                    logger.debug(f"Processing item {idx}: {item.get('name', 'No Title')[:50]}...")
                    # Parse video metadata
                    video = Video(
                        title=item.get("name", "No Title"),
                        url=item.get("url", ""),
                        thumbnail_url=item.get("thumbnailUrl", ""),
                        duration=str(item.get("duration", 0)),
                        video_id=str(item.get("uuid", "")),
                        author=item.get("account", {}).get("name", "Unknown"),
                        published=datetime.fromisoformat(item.get("publishedAt").replace('Z', '+00:00'))
                            if item.get("publishedAt") else None,
                        platform="PeerTube"
                    )
                    videos.append(video)
                    logger.debug(f"Successfully created video: {video.title[:30]}...")
                except Exception as e:
                    logger.error(f"Error parsing video {idx}: {e}")
                    logger.error(f"Problematic item: {item}")
            # Get total results count
            total_results = data.get("total", len(videos))
            logger.info(f"Successfully parsed {len(videos)} videos (total: {total_results})")
            return videos, total_results
        except requests.exceptions.RequestException as e:
            logger.error(f"Network error: {e}")
            return [], 0
        except ValueError as e:
            logger.error(f"JSON parsing error: {e}")
            logger.error(f"Response text: {resp.text[:200] if 'resp' in locals() else 'No response'}")
            return [], 0
        except Exception as e:
            logger.error(f"Unexpected error: {e}")
            logger.exception("Full traceback:")
            return [], 0

    @staticmethod
    @lru_cache(maxsize=32)
    def fetch_trending_videos(
        page: int = 1,
        count: int = DEFAULT_SEARCH_COUNT,
        _cache_key: str = datetime.now().strftime("%Y-%m-%d-%H")  # Cache per hour
    ) -> Tuple[List[Video], int]:
        """
        Fetch trending videos from SepiaSearch API.
        Args:
            page: Page number for pagination.
            count: Number of results per page.
            _cache_key: Internal use for cache busting.
        Returns:
            Tuple of (List[Video], total_results).
        """
        logger.info(f"Fetching trending videos (page {page}, count {count})")
        try:
            params = {
                'start': (page - 1) * count,
                'count': count,
                'nsfw': 'false',
                'sort': '-hot',
                'publishedAt': '-last_30days'
            }
            url = f"{SEPIASEARCH_BASE_URL}?{urlencode(params)}"
            logger.debug(f"Making request to: {url}")
            resp = SearchService.SESSION.get(url, timeout=10)
            resp.raise_for_status()
            data = resp.json()
            videos_data = data.get("data", [])
            # Sort by publication date (newest first)
            videos_data.sort(key=lambda x: x.get("publishedAt", ""), reverse=True)
            videos = []
            for item in videos_data:
                try:
                    video = Video(
                        title=item.get("name", "No Title"),
                        url=item.get("url", ""),
                        thumbnail_url=item.get("thumbnailUrl", ""),
                        duration=str(item.get("duration", 0)),
                        video_id=str(item.get("uuid", "")),
                        author=item.get("account", {}).get("name", "Unknown"),
                        published=datetime.fromisoformat(item.get("publishedAt").replace('Z', '+00:00'))
                            if item.get("publishedAt") else None,
                        platform="PeerTube"
                    )
                    videos.append(video)
                except Exception as e:
                    logger.error(f"Error parsing trending video: {e}")
                    logger.error(f"Problematic item: {item}")
            total_results = data.get("total", len(videos))
            logger.info(f"Successfully fetched {len(videos)} trending videos (total: {total_results})")
            return videos, total_results
        except requests.exceptions.RequestException as e:
            logger.error(f"Network error fetching trending videos: {e}")
            return [], 0
        except Exception as e:
            logger.error(f"Unexpected error fetching trending videos: {e}")
            logger.exception("Full traceback:")
            return [], 0

