#!/usr/bin/env python3

import gi
gi.require_version('Gtk', '3.0')
from gi.repository import Gtk, Gdk

import shlex
import subprocess
import os
import shutil

class PreferencesWindow(Gtk.Dialog):
    """
    A Gtk.Dialog for managing application preferences.
    """
    def __init__(self, parent, preferences):
        """
        Initializes the preferences window with the given parent and preferences.
        
        Args:
            parent (Gtk.Window): The parent window.
            preferences (dict): A dictionary containing the current preferences.
        """
        super().__init__(
            title="✴️ Preferences",
            parent=parent,
            flags=0,
            buttons=(Gtk.STOCK_HELP, Gtk.ResponseType.HELP,
                     Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
                     Gtk.STOCK_APPLY, Gtk.ResponseType.APPLY,
                     Gtk.STOCK_CLOSE, Gtk.ResponseType.CLOSE)
        )
        self.parent = parent
        # Create a copy of the preferences to work with locally
        self.preferences = preferences.copy()
        self.set_default_size(500, 350)

        # Main layout container
        grid = Gtk.Grid(column_spacing=10, row_spacing=10, border_width=12)
        self.get_content_area().add(grid)
        row = 0

        # --- General Options ---
        general_frame = Gtk.Frame(label="✴️ General Options")
        grid.attach(general_frame, 0, row, 2, 1)
        general_grid = Gtk.Grid(column_spacing=10, row_spacing=10, border_width=12)
        general_frame.add(general_grid)

        self.disable_confirm_check = Gtk.CheckButton.new_with_label(
            "Disable confirmation dialogs for actions"
        )
        self.disable_confirm_check.set_active(self.preferences.get("disable_confirm", False))
        general_grid.attach(self.disable_confirm_check, 0, 0, 2, 1)

        self.clear_cache_check = Gtk.CheckButton.new_with_label(
            "Clean DNF cache after successful action"
        )
        self.clear_cache_check.set_active(self.preferences.get("clear_cache", False))
        general_grid.attach(self.clear_cache_check, 0, 1, 2, 1)

        row += 1

        # --- Command Options ---
        command_frame = Gtk.Frame(label="💻️ Command Options")
        grid.attach(command_frame, 0, row, 2, 1)
        command_grid = Gtk.Grid(column_spacing=10, row_spacing=10, border_width=12)
        command_frame.add(command_grid)

        command_label = Gtk.Label(label="💻️ Command Arguments: dnf", xalign=0)
        command_grid.attach(command_label, 0, 0, 1, 1)

        self.command_entry = Gtk.Entry()
        self.command_entry.set_text(self.preferences.get("command", ""))
        command_grid.attach(self.command_entry, 1, 0, 1, 1)

        self.test_button = Gtk.Button(label="✔️ Run")
        # Connect the button to the corrected class method
        self.test_button.connect("clicked", self.on_test_clicked)
        command_grid.attach(self.test_button, 2, 0, 1, 1)

        row += 1

        # --- History Options ---
        history_frame = Gtk.Frame(label="⏳️ History Options")
        grid.attach(history_frame, 0, row, 2, 1)
        history_grid = Gtk.Grid(column_spacing=10, row_spacing=10, border_width=12)
        history_frame.add(history_grid)

        history_size_label = Gtk.Label(label="History Size:", xalign=0)
        history_grid.attach(history_size_label, 0, 0, 1, 1)

        self.history_size_adj = Gtk.Adjustment(
            value=self.preferences.get("history_size", 50),
            lower=1,
            upper=1000,
            step_increment=1,
            page_increment=10
        )
        self.history_size_spin = Gtk.SpinButton(
            adjustment=self.history_size_adj,
            climb_rate=0,
            digits=0
        )
        history_grid.attach(self.history_size_spin, 1, 0, 1, 1)

        # Connect the dialog's response signal to the class method
        self.connect("response", self.on_response)
        self.show_all()

    def on_test_clicked(self, button):
        """
        Handles the "Run" button click event.
        Executes the DNF command and shows the output in a dialog.
        """
        user_args = self.command_entry.get_text().strip()
        if not user_args:
            self.parent.log("No command arguments provided.")
            return

        # Prepend 'dnf' automatically and split arguments
        full_cmd = ["dnf"] + shlex.split(user_args)

        try:
            # Check if dnf exists and is executable
            dnf_path = shutil.which("dnf")
            if not dnf_path or not os.access(dnf_path, os.X_OK):
                output = "Error: 'dnf' command not found or is not executable."
            else:
                # Run the command and capture output
                result = subprocess.run(full_cmd, capture_output=True, text=True, timeout=5)
                output = f"Command: {' '.join(full_cmd)}\nReturn code: {result.returncode}\n\n{result.stdout}\n{result.stderr}"
        except FileNotFoundError:
            output = "Error: 'dnf' command not found. Make sure it is installed."
        except subprocess.TimeoutExpired:
            output = "Error: Command timed out. Check your arguments."
        except Exception as e:
            output = f"Error: {e}"

        # Create a dialog to display the command output
        dialog = Gtk.MessageDialog(
            transient_for=self,
            modal=True,
            message_type=Gtk.MessageType.INFO,
            buttons=Gtk.ButtonsType.CLOSE,
            text="💻️ DNF Command Output"
        )
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        scrolled.set_min_content_height(200)
        scrolled.set_min_content_width(550)

        label = Gtk.Label(label=output)
        label.set_selectable(True)
        label.set_line_wrap(True)
        label.set_xalign(0)
        label.set_yalign(0)
        label.set_hexpand(True)
        label.set_vexpand(True)

        scrolled.add(label)
        content_area = dialog.get_content_area()
        content_area.pack_start(scrolled, True, True, 0)
        content_area.show_all()

        dialog.run()
        dialog.destroy()

    def show_help_dialog(self):
        """
        Displays a help dialog with sample DNF command text.
        """
        help_text = """
🛟️ Common DNF commands:

- -h, --help   Print help
- info [package]: Show detailed information about a package.
- install [package]: Install one or more packages.
- update: Update all installed packages.
- upgrade: Same as update.
- dist-upgrade APT-RPM-like alias for 'distro-sync'
- repoclosure  Print list of unresolved dependencies for repositories
- reposync Synchronize a remote DNF repository to a local directory.
- search [package]: Search for packages.
- remove [package]: Remove one or more packages.
- autoremove: Remove packages installed as dependencies no longer needed.
- list installed: List all installed packages.
- download   Download software to the current directory
- makecache  Generate the metadata cache
- history: Show the transaction history.
- clean all: Remove all cached metadata and packages.
- builddep   Install build dependencies for package or spec file
- changelog  Show package changelogs
- --no-gpgchecks disable OpenPGP signature checking 
- --no-plugins  Disable all libdnf5 plugins
- --color=COLOR  Control whether color is used.
- --refresh  Force refreshing metadata before running the command.
- check-update    Alias for 'check-upgrade'
- dg  Alias for 'downgrade'
- dsync Alias for 'distro-sync'
- erase Alias for 'remove'
- grp   Alias for 'group'
- if    Alias for 'info'
- in    Alias for 'install'
- ls    Alias for 'list'
- mc    Alias for 'makecache'
- rei   Alias for 'reinstall'
- repoinfo    Alias for 'repo info'
- repolist    Alias for 'repo list'
- rm    Alias for 'remove'
- rq    Alias for 'repoquery'
- se    Alias for 'search'
- up    Alias for 'upgrade'
"""

        dialog = Gtk.MessageDialog(
            transient_for=self,
            modal=True,
            message_type=Gtk.MessageType.INFO,
            buttons=Gtk.ButtonsType.CLOSE,
            text="🛟️ DNF Help Commands"
        )
        scrolled = Gtk.ScrolledWindow()
        scrolled.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        scrolled.set_min_content_height(300)
        scrolled.set_min_content_width(550)

        label = Gtk.Label(label=help_text)
        label.set_selectable(True)
        label.set_line_wrap(True)
        label.set_xalign(0)
        label.set_yalign(0)
        label.set_hexpand(True)
        label.set_vexpand(True)

        scrolled.add(label)
        content_area = dialog.get_content_area()
        content_area.pack_start(scrolled, True, True, 0)
        content_area.show_all()

        dialog.run()
        dialog.destroy()

    def on_response(self, dialog, response_id):
        """
        Handles the response from the preferences dialog buttons.
        """
        if response_id == Gtk.ResponseType.HELP:
            self.show_help_dialog()
            return

        if response_id in (Gtk.ResponseType.APPLY, Gtk.ResponseType.CLOSE):
            # Update the local preferences from the UI controls
            self.preferences["disable_confirm"] = self.disable_confirm_check.get_active()
            self.preferences["clear_cache"] = self.clear_cache_check.get_active()
            self.preferences["history_size"] = self.history_size_spin.get_value_as_int()
            self.preferences["command"] = self.command_entry.get_text().strip()

            # Pass the updated preferences back to the parent window
            self.parent.preferences.update(self.preferences)
            self.parent.save_preferences(self.parent.preferences)

            if response_id == Gtk.ResponseType.APPLY:
                self.parent.log("Preferences applied.")
            elif response_id == Gtk.ResponseType.CLOSE:
                self.parent.log("Preferences saved on close.")

        if response_id in (Gtk.ResponseType.CANCEL, Gtk.ResponseType.CLOSE):
            # Destroy the dialog on cancel or close
            dialog.destroy()


class ApplicationWindow(Gtk.Window):
    """
    A simple application window to demonstrate the PreferencesWindow dialog.
    """
    def __init__(self):
        super().__init__(title="Gtk DNF Manager")
        self.set_default_size(800, 600)
        self.set_position(Gtk.WindowPosition.CENTER)
        self.connect("destroy", Gtk.main_quit)

        # Dummy preferences for demonstration
        self.preferences = {
            "disable_confirm": False,
            "clear_cache": True,
            "history_size": 50,
            "command": "--assumeyes"
        }

        # Main layout
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        self.add(vbox)

        # Header bar with preferences button
        header_bar = Gtk.HeaderBar(show_close_button=True, title="Gtk DNF Manager")
        self.set_titlebar(header_bar)
        
        pref_button = Gtk.Button()
        pref_button.set_label("⚙️ Preferences")
        pref_button.connect("clicked", self.on_preferences_clicked)
        header_bar.pack_end(pref_button)

        # Log output area
        self.log_buffer = Gtk.TextBuffer()
        log_view = Gtk.TextView(buffer=self.log_buffer)
        log_view.set_editable(False)
        log_view.set_cursor_visible(False)
        
        scrolled_log = Gtk.ScrolledWindow()
        scrolled_log.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        scrolled_log.add(log_view)
        vbox.pack_start(scrolled_log, True, True, 0)
        
        self.log("Application started. Click 'Preferences' to open the dialog.")
        
    def on_preferences_clicked(self, button):
        """
        Opens the preferences dialog when the button is clicked.
        """
        dialog = PreferencesWindow(self, self.preferences)
        dialog.run()
        dialog.destroy()
        
    def log(self, message):
        """
        A dummy method to log messages to the text view.
        In a real application, this would write to a log file or a more
        robust console.
        """
        end_iter = self.log_buffer.get_end_iter()
        self.log_buffer.insert(end_iter, f"\n{message}")
        
    def save_preferences(self, preferences):
        """
        A dummy method to simulate saving preferences.
        """
        self.log(f"Preferences saved: {preferences}")

def main():
    """
    Main function to run the application.
    """
    win = ApplicationWindow()
    win.show_all()
    Gtk.main()

if __name__ == "__main__":
    main()

