import gi
gi.require_version('Gtk', '3.0')
from gi.repository import Gtk, Gdk, GLib, Pango
import os
import subprocess
import re
import shutil
import datetime
import threading
from repo_selector import RepoSelectorDialog

# New dialog class for the popup message
class MessageDialog(Gtk.Dialog):
    def __init__(self, parent, message):
        Gtk.Dialog.__init__(self, title="Please Wait", parent=parent, flags=Gtk.DialogFlags.MODAL)
        self.set_default_size(300, 50)
        self.set_resizable(False)
        self.set_deletable(False)

        box = self.get_content_area()
        label = Gtk.Label(label=message)
        label.set_margin_top(10)
        label.set_margin_bottom(10)
        box.add(label)
        self.show_all()

# The main dialog class
class RepositoriesDialog(Gtk.Dialog):
    def __init__(self, parent, dnf_cmd, log_func, run_command_func, on_refresh_callback=None):
        Gtk.Dialog.__init__(self, title="Repositories", parent=parent, flags=0)
        self.set_default_size(400, 300)
        
        self.on_refresh_callback = on_refresh_callback
        
        self.dnf_cmd = dnf_cmd
        self.log = log_func
        self.run_command = run_command_func
        
        self.repo_file_path = "/etc/repo.d/default.repo"
        self.changes_made = {}
        self.original_states = {}
        
        self.refresh_dialog = None

        box = self.get_content_area()
        self.grid = Gtk.Grid(column_spacing=10, row_spacing=10)
        self.grid.set_margin_top(10)
        self.grid.set_margin_bottom(10)
        self.grid.set_margin_start(10)
        self.grid.set_margin_end(10)
        box.add(self.grid)

        # The treeview is now the first item in the grid
        self.treeview, self.liststore = self._create_treeview()
        scrolled_window = Gtk.ScrolledWindow()
        scrolled_window.set_vexpand(True)
        scrolled_window.set_hexpand(True)
        scrolled_window.add(self.treeview)
        self.grid.attach(scrolled_window, 0, 0, 2, 1)

        # Add the buttons with icons
        change_button = Gtk.Button(label="Change")
        change_button.set_image(Gtk.Image.new_from_icon_name("document-properties-symbolic", Gtk.IconSize.BUTTON))
        self.add_action_widget(change_button, 102)

        refresh_button = Gtk.Button(label="Refresh")
        refresh_button.set_image(Gtk.Image.new_from_icon_name("view-refresh-symbolic", Gtk.IconSize.BUTTON))
        self.add_action_widget(refresh_button, 101)

        save_button = Gtk.Button(label="Save")
        save_button.set_image(Gtk.Image.new_from_icon_name("document-save-symbolic", Gtk.IconSize.BUTTON))
        self.add_action_widget(save_button, Gtk.ResponseType.OK)

        close_button = Gtk.Button(label="Close")
        close_button.set_image(Gtk.Image.new_from_icon_name("window-close-symbolic", Gtk.IconSize.BUTTON))
        self.add_action_widget(close_button, Gtk.ResponseType.CANCEL)
        
        self.connect("response", self.on_response)
        
        self._load_all_info()
        
        self.show_all()

    def _create_treeview(self):
        store = Gtk.ListStore(str, bool)
        treeview = Gtk.TreeView(model=store)
        
        # Column 1: Repository Name
        renderer_repo_name = Gtk.CellRendererText()
        column_repo_name = Gtk.TreeViewColumn("Name", renderer_repo_name, text=0)
        column_repo_name.set_sizing(Gtk.TreeViewColumnSizing.FIXED)
        column_repo_name.set_fixed_width(250)
        treeview.append_column(column_repo_name)
        
        # Column 2: Enabled/Disabled Toggle
        renderer_toggle = Gtk.CellRendererToggle()
        renderer_toggle.connect("toggled", self.on_toggle_toggled)
        column_enabled = Gtk.TreeViewColumn("Enabled", renderer_toggle, active=1)
        column_enabled.set_sizing(Gtk.TreeViewColumnSizing.FIXED)
        column_enabled.set_fixed_width(80)
        treeview.append_column(column_enabled)

        return treeview, store

    def _load_repositories(self):
        self.liststore.clear()
        self.original_states.clear()
        if not os.path.exists(self.repo_file_path):
            self.log(f"Error: Repository file not found at {self.repo_file_path}. Cannot load repositories.")
            return

        current_repo_name = None
        enabled = 0
        with open(self.repo_file_path, 'r') as f:
            for line in f:
                line = line.strip()
                if line.startswith('['):
                    if current_repo_name:
                        self.liststore.append([current_repo_name, enabled == 1])
                        self.original_states[current_repo_name] = (enabled == 1)
                    current_repo_name = line.strip('[]')
                    enabled = 0
                elif line.startswith("enabled="):
                    enabled = int(line.split('=')[1])

        if current_repo_name:
            self.liststore.append([current_repo_name, enabled == 1])
            self.original_states[current_repo_name] = (enabled == 1)
            
    def _load_current_repo_info(self):
        pass
        
    def _load_all_info(self):
        self._load_current_repo_info()
        self._load_repositories()

    def on_toggle_toggled(self, cell, path):
        store = self.treeview.get_model()
        treeiter = store.get_iter(path)
        if treeiter:
            repo_name = store.get_value(treeiter, 0)
            is_enabled = store.get_value(treeiter, 1)
            new_state = not is_enabled
            
            if new_state != self.original_states.get(repo_name):
                self.changes_made[repo_name] = 1 if new_state else 0
            elif repo_name in self.changes_made:
                del self.changes_made[repo_name]
            
            store.set_value(treeiter, 1, new_state)

    def on_response(self, dialog, response_id):
        if response_id == 102:
            self._launch_repo_selector()
        elif response_id == 101:
            self._on_refresh_button_clicked()
        elif response_id == Gtk.ResponseType.OK:
            if not self.changes_made:
                self.log("No changes to apply. Skipping save.")
                self.destroy()
                return
            
            self._apply_changes()
            self._load_all_info()
        elif response_id == Gtk.ResponseType.CANCEL:
            self.destroy()

    def _apply_changes(self):
        if not self.changes_made:
            self.log("No changes to apply.")
            return

        for repo_name, new_state in self.changes_made.items():
            self.log(f"Applying change for repository '{repo_name}': enabled={new_state}")
            
            sed_command = f"/^\\[{repo_name}\\]/,/^\\[/{{ /enabled=/s/=.*/={new_state}/ }}"
            command = ["sed", "-i", sed_command, self.repo_file_path]
            
            self.run_command(
                command,
                f"Successfully changed state for repository '{repo_name}'.",
                f"Failed to change state for repository '{repo_name}'.",
                use_pkexec=False
            )
        
        self.changes_made.clear()

    def _on_refresh_button_clicked(self):
        self.refresh_dialog = MessageDialog(self, "Refreshing Package List...")
        
        refresh_thread = threading.Thread(target=self._run_dnf_in_thread)
        refresh_thread.start()

    def _run_dnf_in_thread(self):
        self.log("Refreshing and updating DNF packages...")
        
        command = ["dnf", "update", "--refresh", "-y"]
        
        try:
            process = subprocess.Popen(command, text=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            stdout, stderr = process.communicate()
            
            if process.returncode == 0:
                self.log("Packages refreshed and updated successfully.")
            else:
                self.log(f"Failed to update packages: {stderr}")

        except FileNotFoundError:
            self.log("Error: 'dnf' command not found.")
            
        GLib.idle_add(self._on_refresh_complete)

    def _on_refresh_complete(self):
        if self.refresh_dialog:
            self.refresh_dialog.destroy()
        
        self._load_all_info()

        if self.on_refresh_callback:
            self.on_refresh_callback()
        
        return GLib.SOURCE_REMOVE

    def _launch_repo_selector(self):
        self.log("Launching repository selector dialog...")
        selector_dialog = RepoSelectorDialog(self.get_transient_for(), self.log)
        
        response = selector_dialog.run()
        selected_repo_name = None
        if response == Gtk.ResponseType.OK:
            selected_repo_name = selector_dialog.get_selected_repo()

        selector_dialog.destroy()
        
        if selected_repo_name:
            self.log(f"Selected repository: {selected_repo_name}")
            
        self._load_repositories()
        if self.on_refresh_callback:
            self.on_refresh_callback()
