#!/usr/bin/env python3
import gi
gi.require_version('Gtk', '3.0')
from gi.repository import Gtk, Gdk, GLib
import os
import shutil
import subprocess
import re

class RepoSelectorDialog(Gtk.Dialog):
    def __init__(self, parent, log_func):
        Gtk.Dialog.__init__(self, title="Select Repository", parent=parent, flags=0)
        
        self.set_modal(True)
        self.set_transient_for(parent)
        self.connect("delete-event", self.on_delete_event)
        self.set_default_size(400, 100)
        self.set_resizable(False)
        
        self.log = log_func
        self.selected_repo = None

        self.repositories = [
            "https://mirror.aarnet.edu.au",
            "https://mirrors.cicku.me",
            "http://distrib-coffee.ipsl.jussieu.fre",
            "http://ftp.fau.de",
            "http://mirror.internode.on.net",
            "http://ftp.jaist.ac.jp",
            "https://ftp.nluug.nl",
            "http://mirror.math.princeton.edu",
            "http://pclinuxos.c3sl.ufpr.br",
            "http://mirrors.uni-ruse.bg",
            "http://ftp.cc.uoc.gr",
            "http://pclinuxos.mirror.wearetriple.com"
        ]
        
        self.repo_names = [
            "aarnet",
            "cicku",
            "coffee",
            "fau",
            "internode",
            "jaist",
            "nluug",
            "princeton",
            "ufpr",
            "uni-ruse",
            "uoc",
            "wearetriple"
        ]
        
        self.repo_source_dir = "/usr/share/dnf-package-manager/repos/"
        self.repo_dest_path = "/etc/repo.d/default.repo"
        self.repo_dir = "/etc/repo.d/"
        
        box = self.get_content_area()
        
        current_url = self._get_current_repo_url()
        
        label_text = f"Current: {current_url}"
        label = Gtk.Label()
        label.set_markup(label_text)
        label.set_margin_top(10)
        box.add(label)

        self.combo_box = Gtk.ComboBoxText()
        self.combo_box.set_margin_start(10)
        self.combo_box.set_margin_end(10)
        self.combo_box.set_margin_top(5)
        self.combo_box.set_margin_bottom(5)
        box.add(self.combo_box)
        
        for repo in self.repositories:
            self.combo_box.append_text(repo)
        
        current_repo_name = self._get_current_repo_name()
        if current_repo_name:
            try:
                index = self.repo_names.index(current_repo_name)
                self.combo_box.set_active(index)
            except ValueError:
                self.log(f"Warning: Current repository '{current_repo_name}' not in list.")
                self.combo_box.set_active(0)
        else:
            self.combo_box.set_active(0)

        self.add_button("✅ Select", Gtk.ResponseType.OK)
        self.add_button("❌ Cancel", Gtk.ResponseType.CANCEL)
        
        self.connect("response", self.on_response)
        self.show_all()

    def on_delete_event(self, dialog, event):
        return True

    def on_response(self, dialog, response_id):
        if response_id == Gtk.ResponseType.OK:
            active_index = self.combo_box.get_active()
            if active_index >= 0:
                self.selected_repo = self.repo_names[active_index]
                self._disable_other_repos()
                self.copy_repo_file(self.selected_repo)
            else:
                self.log("No online repository selected.")
        
        self.destroy()

    def get_selected_repo(self):
        return self.selected_repo

    def _get_current_repo_name(self):
        if not os.path.exists(self.repo_dest_path):
            return "N/A"
        
        try:
            with open(self.repo_dest_path, 'r') as f:
                content = f.read()
            
            repo_name_match = re.search(r"\[(.+)\]", content)
            if repo_name_match:
                return repo_name_match.group(1)
        except Exception as e:
            self.log(f"Error reading repository name: {e}")
        
        return "N/A"
        
    def _get_current_repo_url(self):
        if not os.path.exists(self.repo_dest_path):
            return "N/A"
            
        try:
            with open(self.repo_dest_path, 'r') as f:
                content = f.read()
            
            url_match = re.search(r"baseurl=(.+)", content)
            if url_match:
                full_url = url_match.group(1).strip()
                
                if full_url.startswith("file:"):
                    local_path = full_url.replace("file://", "").replace("file:", "")
                    return re.sub(r'/pclinuxos/64bit/RPMS.*', '', local_path)
                else:
                    match = re.search(r'https?://[^/]+', full_url)
                    if match:
                        return match.group(0)
                    else:
                        return full_url
        except Exception as e:
            self.log(f"Error reading repository URL: {e}")
            
        return "N/A"

    def _disable_other_repos(self):
        for filename in os.listdir(self.repo_dir):
            if filename.endswith(".repo") and filename != "default.repo":
                filepath = os.path.join(self.repo_dir, filename)
                try:
                    with open(filepath, 'r') as f:
                        content = f.read()
                    
                    new_content = re.sub(r'enabled=1', 'enabled=0', content, flags=re.IGNORECASE)
                    
                    if new_content != content:
                        with open(filepath, 'w') as f:
                            f.write(new_content)
                        self.log(f"Disabled {filename}")
                except Exception as e:
                    self.log(f"Could not disable {filename}: {e}")

    def copy_repo_file(self, repo_name):
        source_file = os.path.join(self.repo_source_dir, f"{repo_name}.repo")
        
        if not os.path.exists(source_file):
            self.log(f"Error: Source file not found: {source_file}")
            return
            
        try:
            shutil.copyfile(source_file, self.repo_dest_path)
            self.log(f"Successfully copied {source_file} to {self.repo_dest_path}.")
        except PermissionError:
            self.log(f"Permission denied. Cannot copy file to {self.repo_dest_path}. Ensure the application is running with root privileges.")
        except Exception as e:
            self.log(f"An unexpected error occurred: {e}")
