#!/usr/bin/env python3

import gi
import subprocess
import os
import sys
import threading

os.environ['GTK_THEME'] = 'Adwaita'  # Forza tema GTK

gi.require_version("Gtk", "3.0")
from gi.repository import Gtk, GdkPixbuf, GLib

class AppInstaller(Gtk.Window):
    def __init__(self):
        Gtk.Window.__init__(self, title="UfficioZero Flatpak Installer")
        self.set_default_size(900, 600)

        self.categories = {
            "Scuola Primaria": [
                {
                    "name": "GCompris",
                    "description": "Suite educativa con oltre 100 attività per imparare giocando.",
                    "icon": "icons/GCompris.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.gcompris",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.gcompris"

                },
                                {
                    "name": "Kalgebra",
                    "description": "Calcolatrice grafica per algebra e funzioni.",
                    "icon": "icons/Kalgebra.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kalgebra",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kalgebra"

                },
                {
                    "name": "Blinken",
                    "description": "Gioco di memoria con sequenze luminose.",
                    "icon": "icons/Blinken.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.blinken",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.blinken"
                },
                {
                    "name": "KBruch",
                    "description": "Esercizi interattivi per imparare e praticare le frazioni.",
                    "icon": "icons/KBruch.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kbruch",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kbruch"
                },
                {
                    "name": "KGeography",
                    "description": "Esplora e impara la geografia con mappe interattive e quiz sui paesi.",
                    "icon": "icons/KGeography.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kgeography",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kgeography"
                },
                {
                    "name": "KLettres",
                    "description": "Impara le lettere e i suoni di diverse lingue in modo semplice e divertente.",
                    "icon": "icons/KLettres.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.klettres",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.klettres"
                },
                {
                    "name": "KTurtle",
                    "description": "Un modo divertente per muovere i primi passi nella programmazione disegnando con la tartaruga.",
                    "icon": "icons/KTurtle.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kturtle",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kturtle"
                },
                {
                    "name": "TuxGuitar",
                    "description": "TuxGuitar è un editor di tablature gratuito e open-source, che consente di creare, modificare e convertire spartiti e tablature per chitarra, supportando diversi formati tra cui Guitar Pro e MIDI.",
                    "icon": "icons/TuxGuitar.png",
                    "command": "flatpak install -y --noninteractive flathub ar.com.tuxguitar.TuxGuitar",
                    "uninstall_command": "flatpak uninstall -y --noninteractive ar.com.tuxguitar.TuxGuitar"
                },
              
                {
                    "name": "MuseScore",
                    "description": "MuseScore è un potente software open source per scrivere, ascoltare, stampare e condividere spartiti musicali in modo facile e professionale.",
                    "icon": "icons/MuseScore.png",
                    "command": "flatpak install -y --noninteractive flathub org.musescore.MuseScore",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.musescore.MuseScore"
                },
                {
                    "name": "Fablemaker",
                    "description": "Entra nel mondo delle favole di Esopo e crea storie interattive personalizzando testi, suoni e immagini.",
                    "icon": "icons/Fablemaker.png",
                    "command": "flatpak install -y --noninteractive flathub com.endlessnetwork.fablemaker",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.endlessnetwork.fablemaker"
                },
                {
                    "name": "The White House",
                    "description": "Scopri i segreti della città colorandola con la programmazione in CSS: impara divertendoti tra oggetti nascosti e indizi da svelare",
                    "icon": "icons/TheWhiteHouse.png",
                    "command": "flatpak install -y --noninteractive flathub com.endlessnetwork.whitehouse",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.endlessnetwork.whitehouse"
                },
                {
                    "name": "Tux Paint",
                    "description": "Tux Paint è un programma di disegno pensato per bambini dai 3 ai 12 anni, con interfaccia semplice, effetti sonori divertenti e un simpatico pinguino guida.",
                    "icon": "icons/TuxPaint.png",
                    "command": "flatpak install -y --noninteractive flathub org.tuxpaint.Tuxpaint",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.tuxpaint.Tuxpaint"
                },
                {
                    "name": "Fretboard",
                    "description": "Scopri e prova accordi di chitarra su un manico interattivo, ideale per principianti e chitarristi esperti.",
                    "icon": "icons/Fretboard.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Fretboard",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Fretboard"
                },
                {
                    "name": "Keypunch",
                    "description": "Allena la digitazione veloce e precisa con esercizi mirati, testi personalizzabili e supporto multilingue.",
                    "icon": "icons/Keypunch.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Keypunch",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Keypunch"
                },
                {
                    "name": "Nootka",
                    "description": "Applicazione interattiva per imparare a leggere le note musicali e suonare spartiti classici, con esercizi personalizzabili, riconoscimento accurato delle melodie in tempo reale e supporto per diversi strumenti musicali.",
                    "icon": "icons/Nootka.png",
                    "command": "flatpak install -y --noninteractive flathub net.sf.nootka",
                    "uninstall_command": "flatpak uninstall -y --noninteractive net.sf.nootka"
                },
            ],

            "Scuola Secondaria di Primo Grado": [
                {
                    "name": "Artikulate",
                    "description": "Allena la pronuncia in lingue straniere confrontando la tua voce con registrazioni di madrelingua.",
                    "icon": "icons/Artikulate.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.artikulate",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.artikulate"
                },
                {
                    "name": "Kalgebra",
                    "description": "Calcolatrice grafica per algebra e funzioni.",
                    "icon": "icons/Kalgebra.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kalgebra",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kalgebra"
                },
                {
                    "name": "KBruch",
                    "description": "Esercizi interattivi per imparare e praticare le frazioni.",
                    "icon": "icons/KBruch.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kbruch",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kbruch"
                },
                {
                    "name": "KGeography",
                    "description": "Esplora e impara la geografia con mappe interattive e quiz sui paesi.",
                    "icon": "icons/KGeography.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kgeography",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kgeography"
                },
                {
                    "name": "Cantor",
                    "description": "Interfaccia grafica per fare calcoli matematici e statistici con software come Python, R, Octave e altri ancora",
                    "icon": "icons/Cantor.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.cantor",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.cantor"
                },
                {
                    "name": "Kalzium",
                    "description": "Tavola periodica interattiva con dati, simulazioni, spettro degli elementi e strumenti per la chimica.",
                    "icon": "icons/Kalzium.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kalzium",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kalzium"
                },
                {
                    "name": "Kig",
                    "description": "Esplora e costruisci figure geometriche interattive per imparare e insegnare la geometria dinamicamente.",
                    "icon": "icons/Kig.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kig",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kig"
                },
                {
                    "name": "Geogebra",
                    "description": "Disegna grafici, risolvi equazioni e costruisci figure dinamiche per scoprire la matematica in modo visivo",
                    "icon": "icons/Geogebra.png",
                    "command": "flatpak install -y --noninteractive flathub org.geogebra.GeoGebra",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.geogebra.GeoGebra"
                },
                {
                    "name": "KStars",
                    "description": "Simula il cielo notturno in tempo reale dal tuo desktop, esplora stelle, pianeti e fenomeni astronomici ovunque ti trovi.",
                    "icon": "icons/KStars.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kstars",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kstars"
                },
                {
                    "name": "Minuet",
                    "description": "Allena l’orecchio musicale con esercizi su intervalli, accordi e scale per migliorare teoria e percezione musicale.",
                    "icon": "icons/Minuet.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.minuet",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.minuet"
                },
                {
                    "name": "MuseScore",
                    "description": "MuseScore è un potente software open source per scrivere, ascoltare, stampare e condividere spartiti musicali in modo facile e professionale.",
                    "icon": "icons/MuseScore.png",
                    "command": "flatpak install -y --noninteractive flathub org.musescore.MuseScore",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.musescore.MuseScore"
                },
                {
                    "name": "KTurtle",
                    "description": "Un modo divertente per muovere i primi passi nella programmazione disegnando con la tartaruga.",
                    "icon": "icons/KTurtle.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kturtle",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kturtle"
                },
                {
                    "name": "TuxGuitar",
                    "description": "TuxGuitar è un editor di tablature gratuito e open-source, che consente di creare, modificare e convertire spartiti e tablature per chitarra, supportando diversi formati tra cui Guitar Pro e MIDI.",
                    "icon": "icons/TuxGuitar.png",
                    "command": "flatpak install -y --noninteractive flathub ar.com.tuxguitar.TuxGuitar",
                    "uninstall_command": "flatpak uninstall -y --noninteractive ar.com.tuxguitar.TuxGuitar"
                },         
                {
                    "name": "Kanagram",
                    "description": "Gioco educativo di anagrammi dove riordinare le lettere per formare parole, con aiuti, livelli personalizzabili e interfaccia adatta per gli alunni.",
                    "icon": "icons/Kanagram.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kanagram",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kanagram"
                },
                {
                    "name": "The Passage",
                    "description": "Un'avventura a scorrimento dove hackerare è la chiave per superare ostacoli e scoprire il bunker nascosto.",
                    "icon": "icons/ThePassage.png",
                    "command": "flatpak install -y --noninteractive flathub com.endlessnetwork.passage",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.endlessnetwork.passage"
                },
                {
                    "name": "Dragon’s Apprentice",
                    "description": "Esplora, risolvi enigmi e risveglia il potere del drago per sconfiggere le forze oscure.",
                    "icon": "icons/DragonsApprentice.png",
                    "command": "flatpak install -y --noninteractive flathub com.endlessnetwork.dragonsapprentice",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.endlessnetwork.dragonsapprentice"
                },
                {
                    "name": "Fretboard",
                    "description": "Scopri e prova accordi di chitarra su un manico interattivo, ideale per principianti e chitarristi esperti.",
                    "icon": "icons/Fretboard.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Fretboard",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Fretboard"
                },
                {
                    "name": "Keypunch",
                    "description": "Allena la digitazione veloce e precisa con esercizi mirati, testi personalizzabili e supporto multilingue.",
                    "icon": "icons/Keypunch.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Keypunch",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Keypunch"
                },
                {
                    "name": "Nootka",
                    "description": "Applicazione interattiva per imparare a leggere le note musicali e suonare spartiti classici, con esercizi personalizzabili, riconoscimento accurato delle melodie in tempo reale e supporto per diversi strumenti musicali.",
                    "icon": "icons/Nootka.png",
                    "command": "flatpak install -y --noninteractive flathub net.sf.nootka",
                    "uninstall_command": "flatpak uninstall -y --noninteractive net.sf.nootka"
                },
                
            ],

            "Scuola Secondaria di Secondo Grado": [
                
                {
                    "name": "Artikulate",
                    "description": "Allena la pronuncia in lingue straniere confrontando la tua voce con registrazioni di madrelingua.",
                    "icon": "icons/Artikulate.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.artikulate",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.artikulate"
                },
               
                {
                    "name": "Kalgebra",
                    "description": "Calcolatrice grafica per algebra e funzioni.",
                    "icon": "icons/Kalgebra.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kalgebra",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kalgebra"
                },
                {
                    "name": "KGeography",
                    "description": "Esplora e impara la geografia con mappe interattive e quiz sui paesi.",
                    "icon": "icons/KGeography.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kgeography",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kgeography"
                },
                {
                    "name": "Avogadro",
                    "description": "Avogadro è un editor molecolare open source avanzato per modellare e visualizzare strutture chimiche, ideale per chimica computazionale, bioinformatica e scienza dei materiali.",
                    "icon": "icons/Avogadro.png",
                    "command": "flatpak install -y --noninteractive flathub org.openchemistry.Avogadro2",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.openchemistry.Avogadro2"
                },
                {
                    "name": "Cantor",
                    "description": "Interfaccia grafica per fare calcoli matematici e statistici con software come Python, R, Octave e altri ancora",
                    "icon": "icons/Cantor.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.cantor",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.cantor"
                },
                {
                    "name": "Kalzium",
                    "description": "Tavola periodica interattiva con dati, simulazioni, spettro degli elementi e strumenti per la chimica.",
                    "icon": "icons/Kalzium.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kalzium",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kalzium"
                },
                {
                    "name": "LibreCAD",
                    "description": "Crea progetti tecnici in 2D con precisione: LibreCAD è l'app CAD open source ideale per piante, componenti e disegni tecnici.",
                    "icon": "icons/LibreCAD.png",
                    "command": "flatpak install -y --noninteractive flathub org.librecad.librecad",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.librecad.librecad"
                },
                {
                    "name": "Kig",
                    "description": "Esplora e costruisci figure geometriche interattive per imparare e insegnare la geometria dinamicamente.",
                    "icon": "icons/Kig.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kig",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kig"
                },
                {
                    "name": "Geogebra",
                    "description": "Disegna grafici, risolvi equazioni e costruisci figure dinamiche per scoprire la matematica in modo visivo",
                    "icon": "icons/Geogebra.png",
                    "command": "flatpak install -y --noninteractive flathub org.geogebra.GeoGebra",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.geogebra.GeoGebra"
                },
                {
                    "name": "Minuet",
                    "description": "Allena l’orecchio musicale con esercizi su intervalli, accordi e scale per migliorare teoria e percezione musicale.",
                    "icon": "icons/Minuet.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.minuet",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.minuet"
                },
                {
                    "name": "KmPlot",
                    "description": "Strumento versatile per disegnare, analizzare e personalizzare grafici di funzioni matematiche.",
                    "icon": "icons/KmPlot.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kmplot",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kmplot"
                },
                {
                    "name": "LabPlot",
                    "description": "Software open source per la creazione di grafici e l’analisi dei dati.",
                    "icon": "icons/LabPlot.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.labplot2",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.labplot2"
                },
                {
                    "name": "The Passage",
                    "description": "Un'avventura a scorrimento dove hackerare è la chiave per superare ostacoli e scoprire il bunker nascosto.",
                    "icon": "icons/ThePassage.png",
                    "command": "flatpak install -y --noninteractive flathub com.endlessnetwork.passage",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.endlessnetwork.passage"
                },
                {
                    "name": "Dragon’s Apprentice",
                    "description": "Esplora, risolvi enigmi e risveglia il potere del drago per sconfiggere le forze oscure.",
                    "icon": "icons/DragonsApprentice.png",
                    "command": "flatpak install -y --noninteractive flathub com.endlessnetwork.dragonsapprentice",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.endlessnetwork.dragonsapprentice"
                },
                {
                    "name": "Fretboard",
                    "description": "Scopri e prova accordi di chitarra su un manico interattivo, ideale per principianti e chitarristi esperti.",
                    "icon": "icons/Fretboard.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Fretboard",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Fretboard"
                },
                {
                    "name": "KTurtle",
                    "description": "Un modo divertente per muovere i primi passi nella programmazione disegnando con la tartaruga.",
                    "icon": "icons/KTurtle.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kturtle",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kturtle"
                },
                {
                    "name": "Keypunch",
                    "description": "Allena la digitazione veloce e precisa con esercizi mirati, testi personalizzabili e supporto multilingue.",
                    "icon": "icons/Keypunch.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Keypunch",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Keypunch"
                },
            ],

            "Università": [
                {
                    "name": "Gramps",
                    "description": "Organizza e analizza ricerche genealogiche con strumenti professionali per tracciare persone, eventi e relazioni familiari in modo dettagliato.",
                    "icon": "icons/Gramps.png",
                    "command": "flatpak install -y --noninteractive flathub org.gramps_project.Gramps",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.gramps_project.Gramps"
                },
                {
                    "name": "Gephi",
                    "description": "Visualizza e studia reti complesse, come relazioni sociali o dati scientifici, con strumenti di analisi avanzati.",
                    "icon": "icons/Gephi.png",
                    "command": "flatpak install -y --noninteractive flathub org.gephi.Gephi",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.gephi.Gephi"
                },
                {
                    "name": "Tropy",
                    "description": "Gestisci e descrivi le tue fonti fotografiche d’archivio in modo organizzato ed efficiente, con uno strumento pensato per ricercatori.",
                    "icon": "icons/Tropy.png",
                    "command": "flatpak install -y --noninteractive flathub org.tropy.Tropy",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.tropy.Tropy"
                },
                {
                    "name": "Geogebra",
                    "description": "Disegna grafici, risolvi equazioni e costruisci figure dinamiche per scoprire la matematica in modo visivo",
                    "icon": "icons/Geogebra.png",
                    "command": "flatpak install -y --noninteractive flathub org.geogebra.GeoGebra",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.geogebra.GeoGebra"
                },
                {
                    "name": "Zotero",
                    "description": " Gestisci e condividi le tue fonti di ricerca con Zotero: raccogli, organizza, cita e collabora in modo semplice ed efficace.",
                    "icon": "icons/Zotero.png",
                    "command": "flatpak install -y --noninteractive flathub org.zotero.Zotero",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.zotero.Zotero"
                },
                {
                    "name": "Alexandria",
                    "description": "Lettore minimalista per EPUB, Kindle e fumetti, con segnalibri, temi e annotazioni.",
                    "icon": "icons/Alexandria.png",
                    "command": "flatpak install -y --noninteractive flathub io.github.btpf.alexandria",
                    "uninstall_command": "flatpak uninstall -y --noninteractive io.github.btpf.alexandria"
                },
                {
                    "name": "Avogadro",
                    "description": "Avogadro è un editor molecolare open source avanzato per modellare e visualizzare strutture chimiche, ideale per chimica computazionale, bioinformatica e scienza dei materiali.",
                    "icon": "icons/Avogadro.png",
                    "command": "flatpak install -y --noninteractive flathub org.openchemistry.Avogadro2",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.openchemistry.Avogadro2"
                },
                {
                    "name": "Code::Blocks",
                    "description": "Ambiente di sviluppo integrato per C, C++ e Fortran, estendibile e compatibile con più compilatori",
                    "icon": "icons/CodeBlocks.png",
                    "command": "flatpak install -y --noninteractive flathub org.codeblocks.codeblocks",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.codeblocks.codeblocks"
                },
                {
                    "name": "KmPlot",
                    "description": "Strumento versatile per disegnare, analizzare e personalizzare grafici di funzioni matematiche.",
                    "icon": "icons/KmPlot.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kmplot",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kmplot"
                },
                {
                    "name": "LabPlot",
                    "description": "Software open source per la creazione di grafici e l’analisi dei dati.",
                    "icon": "icons/LabPlot.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.labplot2",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.labplot2"
                },
                {
                    "name": "GPS Correlate",
                    "description": "GPS Correlate aggiunge automaticamente le coordinate GPS alle foto, sincronizzando l'orario dello scatto con i dati di tracciamento di un dispositivo GPS.",
                    "icon": "icons/GPSCorrelate.png",
                    "command": "flatpak install -y --noninteractive flathub io.github.dfandrich.gpscorrelate",
                    "uninstall_command": "flatpak uninstall -y --noninteractive io.github.dfandrich.gpscorrelate"
                },
                {
                    "name": "Manuskript",
                    "description": "Ambiente di scrittura open-source per sviluppare trame, personaggi e capitoli in modo strutturato, con modalità senza distrazioni e strumenti di esportazione.",
                    "icon": "icons/Manuskript.png",
                    "command": "flatpak install -y --noninteractive flathub ch.theologeek.Manuskript",
                    "uninstall_command": "flatpak uninstall -y --noninteractive ch.theologeek.Manuskript"
                },
                {
                    "name": "Texmaker",
                    "description": "Editor LaTeX completo con anteprima PDF integrata, evidenziazione della sintassi e strumenti matematici.",
                    "icon": "icons/Texmaker.png",
                    "command": "flatpak install -y --noninteractive flathub net.xm1math.Texmaker",
                    "uninstall_command": "flatpak uninstall -y --noninteractive net.xm1math.Texmaker"
                },
                {
                    "name": "GNUOctave",
                    "description": "Ambiente interattivo per calcoli numerici avanzati: GNU Octave è un potente linguaggio simile a Matlab per algebra lineare, grafici, equazioni differenziali e molto altro.",
                    "icon": "icons/GNUOctave.png",
                    "command": "flatpak install -y --noninteractive flathub org.octave.Octave",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.octave.Octave"
                },
                {
                    "name": "Geany",
                    "description": " IDE veloce e leggero con evidenziazione della sintassi, completamento automatico e supporto per molti linguaggi, ideale per ambienti minimali.",
                    "icon": "icons/Geany.png",
                    "command": "flatpak install -y --noninteractive flathub org.geany.Geany",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.geany.Geany"
                },
                {
                    "name": "Kalgebra",
                    "description": "Calcolatrice grafica per algebra e funzioni.",
                    "icon": "icons/Kalgebra.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kalgebra",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kalgebra"
                },
                {
                    "name": "Jabref",
                    "description": "JabRef è un'applicazione open source per la gestione di database bibliografici BibTeX e biblatex: ideale per ricercatori, studenti e accademici che vogliono ordinare, citare e trovare facilmente le proprie fonti.",
                    "icon": "icons/Jabref.png",
                    "command": "flatpak install -y --noninteractive flathub org.jabref.jabref",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.jabref.jabref"
                },
                {
                    "name": "Cantor",
                    "description": "Interfaccia grafica per fare calcoli matematici e statistici con software come Python, R, Octave e altri ancora",
                    "icon": "icons/Cantor.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.cantor",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.cantor"
                },
                {
                    "name": "Qucs",
                    "description": "Qucs-S è un simulatore di circuiti con interfaccia grafica che supporta SPICE e altri motori di simulazione, ideale per progettare e analizzare circuiti elettronici.",
                    "icon": "icons/Qucs.png",
                    "command": "flatpak install -y --noninteractive flathub io.github.ra3xdh.qucs_s",
                    "uninstall_command": "flatpak uninstall -y --noninteractive io.github.ra3xdh.qucs_s"
                },
                {
                    "name": "PyMOL",
                    "description": "Visualizza e analizza strutture molecolari in 3D con PyMOL, lo strumento open source ideale per la biologia strutturale e la generazione di immagini scientifiche di alta qualità.",
                    "icon": "icons/PyMOL.png",
                    "command": "flatpak install -y --noninteractive flathub org.pymol.PyMOL",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.pymol.PyMOL"
                },
                
                {
                    "name": "LibreCAD",
                    "description": "Crea progetti tecnici in 2D con precisione: LibreCAD è l'app CAD open source ideale per piante, componenti e disegni tecnici.",
                    "icon": "icons/LibreCAD.png",
                    "command": "flatpak install -y --noninteractive flathub org.librecad.librecad",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.librecad.librecad"
                },
                {
                    "name": "Rocs",
                    "description": "IDE per la teoria dei grafi: con Rocs puoi creare, visualizzare e analizzare algoritmi su grafi in modo intuitivo usando JavaScript, ideale per studenti e appassionati.",
                    "icon": "icons/Rocs.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.rocs",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.rocs"
                },
                {
                    "name": "Freeplane",
                    "description": "Crea mappe mentali per organizzare idee, gestire progetti e visualizzare conoscenze in modo chiaro ed efficace con Freeplane",
                    "icon": "icons/Freeplane.png",
                    "command": "flatpak install -y --noninteractive flathub org.freeplane.App",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.freeplane.App"
                },
                {
                    "name": "gImageReader",
                    "description": "Interfaccia grafica per Tesseract-OCR: importa immagini e PDF, riconosce il testo e lo esporta in vari formati, con supporto a scansioni, spell check e creazione di documenti ODT o PDF.",
                    "icon": "icons/gImageReader.png",
                    "command": "flatpak install -y --noninteractive flathub io.github.manisandro.gImageReader",
                    "uninstall_command": "flatpak uninstall -y --noninteractive io.github.manisandro.gImageReader"
                },
                {
                    "name": "OCRFeeder",
                    "description": "Suite completa per OCR e riconoscimento del layout: analizza immagini e PDF, rileva testi e grafica, e li trasforma in documenti modificabili con un'interfaccia GTK intuitiva.",
                    "icon": "icons/OCRFeeder.png",
                    "command": "flatpak install -y --noninteractive flathub org.gnome.OCRFeeder",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.gnome.OCRFeeder"
                },
                {
                    "name": "Kalzium",
                    "description": "Tavola periodica interattiva con dati, simulazioni, spettro degli elementi e strumenti per la chimica.",
                    "icon": "icons/Kalzium.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kalzium",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kalzium"
                },
                {
                    "name": "VSCodium",
                    "description": "Ambiente di sviluppo open source e rispettoso della privacy, basato su Visual Studio Code ma senza tracciamenti.",
                    "icon": "icons/VSCodium.png",
                    "command": "flatpak install -y --noninteractive flathub com.vscodium.codium",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.vscodium.codium"
                },
                {
                    "name": "Scilab",
                    "description": "Ambiente scientifico completo per calcoli numerici: Scilab offre strumenti per simulazione, visualizzazione 2D/3D, controllo, statistica e modellazione di sistemi dinamici con Xcos.",
                    "icon": "icons/Scilab.png",
                    "command": "flatpak install -y --noninteractive flathub org.scilab.Scilab",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.scilab.Scilab"
                },
                {
                    "name": "GNU PSPP",
                    "description": "Alternativa libera a SPSS per l’analisi statistica di dati campionari, PSPP supporta regressioni, test, ANOVA e molto altro, senza limiti o licenze da acquistare.",
                    "icon": "icons/GNUPSPP.png",
                    "command": "flatpak install -y --noninteractive flathub org.gnu.pspp",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.gnu.pspp"
                },
                {
                    "name": "KBibTeX",
                    "description": "Gestisci le tue bibliografie LaTeX con facilità: KBibTeX consente di modificare, importare, esportare e visualizzare citazioni in diversi formati, con integrazione a editor LaTeX e database online",
                    "icon": "icons/KBibTeX.png",
                    "command": "flatpak install -y --noninteractive flathub org.kde.kbibtex",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.kde.kbibtex"
                },
                {
                    "name": "Fretboard",
                    "description": "Scopri e prova accordi di chitarra su un manico interattivo, ideale per principianti e chitarristi esperti.",
                    "icon": "icons/Fretboard.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Fretboard",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Fretboard"
                },
                {
                    "name": "Keypunch",
                    "description": "Allena la digitazione veloce e precisa con esercizi mirati, testi personalizzabili e supporto multilingue.",
                    "icon": "icons/Keypunch.png",
                    "command": "flatpak install -y --noninteractive flathub dev.bragefuglseth.Keypunch",
                    "uninstall_command": "flatpak uninstall -y --noninteractive dev.bragefuglseth.Keypunch"
                },
                {
                    "name": "Zettlr",
                    "description": "Zettlr è un editor Markdown moderno, flessibile e open source, ideale per scrittori, ricercatori e giornalisti: supporta Zettelkasten, citazioni con Zotero e BibTeX, temi personalizzabili e un potente motore di ricerca interno",
                    "icon": "icons/Zettlr.png",
                    "command": "flatpak install -y --noninteractive flathub com.zettlr.Zettlr",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.zettlr.Zettlr"
                },
            ],

            "Strumenti per docenti": [
                {
                    "name": "OpenBoard",
                    "description": "OpenBoard è una lavagna interattiva open source per scuole e università, ideale per l’insegnamento con LIM o in configurazioni a doppio schermo con penna e proiettore.",
                    "icon": "icons/OpenBoard.png",
                    "command": "flatpak install -y --noninteractive flathub ch.openboard.OpenBoard",
                    "uninstall_command": "flatpak uninstall -y --noninteractive ch.openboard.OpenBoard"
                },
                {
                    "name": "Freeplane",
                    "description": "Crea mappe mentali per organizzare idee, gestire progetti e visualizzare conoscenze in modo chiaro ed efficace con Freeplane",
                    "icon": "icons/Freeplane.png",
                    "command": "flatpak install -y --noninteractive flathub org.freeplane.App",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.freeplane.App"
                },
                {
                    "name": "Solanum",
                    "description": "Solanum ti aiuta a bilanciare lavoro e pause usando la tecnica del pomodoro: 4 sessioni di lavoro intervallate da pause brevi e una lunga.",
                    "icon": "icons/Solanum.png",
                    "command": "flatpak install -y --noninteractive flathub org.gnome.Solanum",
                    "uninstall_command": "flatpak uninstall -y --noninteractive org.gnome.Solanum"
                },
                
                {
                    "name": "Zettlr",
                    "description": "Zettlr è un editor Markdown moderno, flessibile e open source, ideale per scrittori, ricercatori e giornalisti: supporta Zettelkasten, citazioni con Zotero e BibTeX, temi personalizzabili e un potente motore di ricerca interno",
                    "icon": "icons/Zettlr.png",
                    "command": "flatpak install -y --noninteractive flathub com.zettlr.Zettlr",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.zettlr.Zettlr"
                },
                               
                {
                    "name": "SpeechNote",
                    "description": "SpeechNote ti consente di prendere, leggere e tradurre le note in più lingue. Utilizza il linguaggio per il testo, il testo in linguaggio e la traduzione automatica per farlo. L'elaborazione del testo e della voce avviene interamente offline, localmente sul tuo Computer, senza utilizzare una connessione di rete. La tua privacy è sempre - rispettato. Nessun dato viene inviato a Internet.",
                    "icon": "icons/SpeechNote.png",
                    "command": "flatpak install -y --noninteractive flathub net.mkiol.SpeechNote",
                    "uninstall_command": "flatpak uninstall -y --noninteractive net.mkiol.SpeechNote"
                },
                
                {
                    "name": "Rustdesk",
                    "description": "RustDesk è una soluzione open source per il controllo remoto sicuro e multipiattaforma, con connessioni P2P cifrate e possibilità di auto-hosting dei dati.",
                    "icon": "icons/Rustdesk.png",
                    "command": "flatpak install -y --noninteractive flathub com.rustdesk.RustDesk",
                    "uninstall_command": "flatpak uninstall -y --noninteractive com.rustdesk.RustDesk"
                    
                },
            ]
        }

        # Liste per tracciare le azioni basate sul cambiamento dello switch
        self.pending_install_actions = {} # app_id: command
        self.pending_uninstall_actions = {} # app_id: command

        self.action_queue = [] # Coda per le operazioni da eseguire
        self.current_category = "Scuola Primaria"
        self.checkboxes = []

        self.build_ui()

    def build_ui(self):
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6)
        self.add(vbox)

        category_bar = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6)
        for cat in self.categories.keys():
            button = Gtk.Button(label=cat)
            button.connect("clicked", self.change_category, cat)
            category_bar.pack_start(button, False, False, 0)
        vbox.pack_start(category_bar, False, False, 0)

        self.scrolled = Gtk.ScrolledWindow()
        self.flowbox = Gtk.FlowBox()
        self.flowbox.set_max_children_per_line(4)
        self.flowbox.set_selection_mode(Gtk.SelectionMode.NONE)
        self.scrolled.add(self.flowbox)
        vbox.pack_start(self.scrolled, True, True, 0)

        self.progress = Gtk.ProgressBar()
        vbox.pack_start(self.progress, False, False, 0)

        self.action_button = Gtk.Button(label="Nessuna Azione")
        self.action_button.set_sensitive(False)
        self.action_button.connect("clicked", self.execute_pending_actions)
        vbox.pack_start(self.action_button, False, False, 5)

        self.load_apps(self.current_category)

    def change_category(self, widget, category):
        self.current_category = category
        self.load_apps(category)

    def get_app_id_from_command(self, command_str):
        # Estrae l'ID dell'app, che è tipicamente l'ultima parte del comando
        return command_str.split()[-1]

    def is_app_installed(self, app_id):
        try:
            subprocess.run(['flatpak', 'info', app_id], check=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            return True
        except (subprocess.CalledProcessError, FileNotFoundError):
            # FileNotFoundError se flatpak non è installato, CalledProcessError se l'app non è trovata
            return False


    def load_apps(self, category):
        ICON_BASE_PATH = "/usr/share/eduflatpak/icons"
    
        for child in self.flowbox.get_children():
            self.flowbox.remove(child)

        self.checkboxes = []

        for app in self.categories[category]:
            app_id = self.get_app_id_from_command(app["command"])
            app['app_id'] = app_id # Memorizza l'ID per un facile accesso
            app['is_installed'] = self.is_app_installed(app_id) # Stato di installazione attuale
            icon_filename = os.path.basename(app["icon"])		
            icon_path = os.path.join(ICON_BASE_PATH, icon_filename)
            
            if not os.path.exists(icon_path):
            	icon_path = os.path.join(ICON_BASE_PATH, "default_icon.png")

            try:
                pixbuf = GdkPixbuf.Pixbuf.new_from_file_at_size(icon_path, 96, 96)
            except Exception as e:
                print(f"Errore caricamento icona {icon_path}: {e}")
                pixbuf = GdkPixbuf.Pixbuf.new_from_file_at_size("icons/default_icon.png", 96, 96)

            image = Gtk.Image.new_from_pixbuf(pixbuf)
            name_label = Gtk.Label(label=app["name"])
            name_label.set_xalign(0.5)
            desc_label = Gtk.Label(label=app["description"])
            desc_label.set_line_wrap(True)
            desc_label.set_xalign(0.5)
            desc_label.set_max_width_chars(30)

            toggle = Gtk.Switch()
            toggle.set_active(app['is_installed']) # Imposta lo switch in base allo stato di installazione

            toggle.app_data = app
            toggle.connect("notify::active", self.on_toggle_changed)

            self.checkboxes.append(toggle)

            toggle_box = Gtk.Box(orientation=Gtk.Orientation.HORIZONTAL, spacing=6, halign=Gtk.Align.CENTER)
            toggle_box.pack_start(toggle, False, False, 0) # Solo lo switch, centrato

            app_box = Gtk.Box(orientation=Gtk.Orientation.VERTICAL, spacing=6, margin=10)
            app_box.set_size_request(200, 220)
            app_box.pack_start(image, False, False, 0)
            app_box.pack_start(name_label, False, False, 0)
            app_box.pack_start(desc_label, False, False, 0)
            app_box.pack_start(toggle_box, False, False, 0)

            frame = Gtk.Frame()
            frame.add(app_box)
            self.flowbox.add(frame)

        self.show_all()

    def on_toggle_changed(self, switch, gparam):
        app_data = switch.app_data
        app_id = app_data['app_id']
        is_currently_active = switch.get_active() # Nuovo stato dello switch
        was_initially_installed = app_data['is_installed'] # Stato quando la UI è stata caricata

        if is_currently_active: # L'utente vuole che l'app sia presente (installata)
            if not was_initially_installed: # Non era installata, ora l'utente la vuole -> INSTALLA
                self.pending_install_actions[app_id] = app_data["command"]
                if app_id in self.pending_uninstall_actions:
                    del self.pending_uninstall_actions[app_id] # Annulla una precedente disinstallazione
            else: # Era già installata, l'utente la lascia attiva -> NESSUNA AZIONE PENDENTE (o annulla disinstallazione)
                if app_id in self.pending_uninstall_actions:
                    del self.pending_uninstall_actions[app_id]
        else: # L'utente NON vuole che l'app sia presente (disinstallata)
            if was_initially_installed: # Era installata, ora l'utente non la vuole -> DISINSTALLA
                if "uninstall_command" in app_data:
                    self.pending_uninstall_actions[app_id] = app_data["uninstall_command"]
                    if app_id in self.pending_install_actions:
                        del self.pending_install_actions[app_id] # Annulla una precedente installazione
            else: # Non era installata, l'utente la lascia disattiva -> NESSUNA AZIONE PENDENTE (o annulla installazione)
                if app_id in self.pending_install_actions:
                    del self.pending_install_actions[app_id]
        
        self.update_action_button_label_and_state()

    def update_action_button_label_and_state(self):
        num_install = len(self.pending_install_actions)
        num_uninstall = len(self.pending_uninstall_actions)

        if num_install > 0 and num_uninstall > 0:
            self.action_button.set_label(f"Applica {num_install + num_uninstall} Modifiche")
            self.action_button.set_sensitive(True)
        elif num_install > 0:
            self.action_button.set_label(f"Installa {num_install} App")
            self.action_button.set_sensitive(True)
        elif num_uninstall > 0:
            self.action_button.set_label(f"Disinstalla {num_uninstall} App")
            self.action_button.set_sensitive(True)
        else:
            self.action_button.set_label("Nessuna Azione")
            self.action_button.set_sensitive(False)

    def execute_pending_actions(self, widget):
        self.action_queue = []
        for app_id, command in self.pending_uninstall_actions.items():
            self.action_queue.append({'command': command, 'type': 'Disinstallazione', 'app_id': app_id})
        for app_id, command in self.pending_install_actions.items():
            self.action_queue.append({'command': command, 'type': 'Installazione', 'app_id': app_id})

        if not self.action_queue:
            self.show_dialog("Nessuna operazione da eseguire.")
            return

        self.pending_install_actions.clear()
        self.pending_uninstall_actions.clear()
        self.run_next_action_in_queue()

    def run_next_action_in_queue(self):
        if not self.action_queue:
            self.progress.set_fraction(0)
            self.close_wait_dialog()
            self.show_dialog("Tutte le operazioni sono state completate.")
            self.load_apps(self.current_category) # Ricarica per aggiornare lo stato degli switch
            self.update_action_button_label_and_state() # Resetta il pulsante
            return
        current_action = self.action_queue.pop(0)
        command = current_action['command']
        action_type_display = current_action['type']
        app_id_display = current_action['app_id']
        self.progress.set_fraction(0.1)
        self.show_wait_dialog(f"{action_type_display} di {app_id_display} in corso...")
        GLib.timeout_add(100, self.process_command_from_queue, command)

    def process_command_from_queue(self, command):
        process = subprocess.Popen(
            command,
            shell=True,
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT,
            text=True
        )

        def read_output():
            for line in iter(process.stdout.readline, ''):
                print(line.strip())
                GLib.idle_add(self.progress.pulse)
            process.stdout.close()
            process.wait()
            GLib.idle_add(self.close_wait_dialog)
            GLib.idle_add(self.run_next_action_in_queue)

        threading.Thread(target=read_output).start()
        return False

    def show_dialog(self, message):
        dialog = Gtk.MessageDialog(parent=self, flags=0,
                                   message_type=Gtk.MessageType.INFO,
                                   buttons=Gtk.ButtonsType.OK,
                                   text=message)
        dialog.run()
        dialog.destroy()

    def show_wait_dialog(self, message):
        self.wait_dialog = Gtk.Dialog(title="Attendere", parent=self, flags=0)
        self.wait_dialog.set_modal(True)
        self.wait_dialog.set_default_size(300, 100)

        box = self.wait_dialog.get_content_area()
        label = Gtk.Label(label=message)
        spinner = Gtk.Spinner()
        spinner.start()

        box.pack_start(label, True, True, 10)
        box.pack_start(spinner, True, True, 10)

        self.wait_dialog.show_all()

    def close_wait_dialog(self):
        if hasattr(self, 'wait_dialog'):
            self.wait_dialog.destroy()
            del self.wait_dialog

if __name__ == "__main__":
    

    win = AppInstaller()
    win.connect("destroy", Gtk.main_quit)
    win.show_all()
    Gtk.main()
