"""Utility functions"""

import subprocess
import sys
from config.settings import API_RATE_LIMITER


def check_dependencies():
    """Check if required dependencies are installed"""
    try:
        # Check yt-dlp
        subprocess.run(["yt-dlp", "--version"], 
                      capture_output=True, check=True)
        
        # Check VLC (optional)
        try:
            import vlc
        except ImportError:
            print("Warning: VLC not available")
            
        return True
        
    except (subprocess.CalledProcessError, FileNotFoundError):
        print("Error: yt-dlp is required but not found")
        print("Install with: pip install yt-dlp")
        return False

def format_duration(seconds: str) -> str:
    """Format duration in seconds to readable format"""
    try:
        sec = int(seconds)
        minutes, seconds = divmod(sec, 60)
        hours, minutes = divmod(minutes, 60)
        
        if hours > 0:
            return f"{hours}:{minutes:02d}:{seconds:02d}"
        else:
            return f"{minutes}:{seconds:02d}"
    except ValueError:
        return "0:00"

def extract_channel_id(video_url: str) -> str:
    """
    Extract channel ID from various video URL formats.
    
    Args:
        video_url: URL of the video
        
    Returns:
        Channel ID string or empty string if not found
    """
    import re
    
    # Placeholder implementation - you'll need to adapt this based on your URL patterns
    patterns = [
        r'/channel/([^/?]+)',
        r'/user/([^/?]+)',
        r'/c/([^/?]+)',
    ]
    
    for pattern in patterns:
        match = re.search(pattern, video_url)
        if match:
            return match.group(1)
    
    return ""


def generate_rss_url(channel_id: str, platform: str = "youtube") -> str:
    """
    Generate RSS feed URL for a channel.
    
    Args:
        channel_id: Channel ID
        platform: Video platform ("youtube", "peertube", etc.)
        
    Returns:
        RSS feed URL string
    """
    if platform == "youtube":
        # YouTube RSS feed URL format
        return f"https://www.youtube.com/feeds/videos.xml?channel_id={channel_id}"
    elif platform == "peertube":
        # PeerTube RSS feed URL format
        return f"https://sepiasearch.org/feeds/videos.xml?videoChannelId={channel_id}"
    else:
        return f"https://example.com/feed/{channel_id}"


def format_duration(seconds: int) -> str:
    """
    Format duration in seconds to HH:MM:SS or MM:SS format.
    
    Args:
        seconds: Duration in seconds
        
    Returns:
        Formatted duration string
    """
    if not seconds:
        return "00:00"
    
    hours = seconds // 3600
    minutes = (seconds % 3600) // 60
    secs = seconds % 60
    
    if hours > 0:
        return f"{hours:02d}:{minutes:02d}:{secs:02d}"
    else:
        return f"{minutes:02d}:{secs:02d}"


def sanitize_filename(filename: str) -> str:
    """
    Sanitize filename by removing invalid characters.
    
    Args:
        filename: Original filename
        
    Returns:
        Sanitized filename
    """
    import re
    # Remove characters that are invalid in filenames
    return re.sub(r'[<>:"/\\|?*]', '_', filename)

def extract_channel_feed_url(video_url: str) -> str:
    """
    Extract channel RSS feed URL from video URL.
    This combines extract_channel_id and generate_rss_url.
    
    Args:
        video_url: URL of the video
        
    Returns:
        RSS feed URL string or empty string if cannot extract
    """
    channel_id = extract_channel_id(video_url)
    if channel_id:
        return generate_rss_url(channel_id)
    return ""
