#!/usr/bin/env python3
"""
PurpleTube Player - Enhanced Modular Version with Queue Support
Maintains all original functionality with better organization and queue integration
"""
import sys
import os
import logging
from pathlib import Path

# Add project root to Python path
project_root = Path(__file__).parent
sys.path.insert(0, str(project_root))

print("🐍 [DEBUG] Python path configuration:")
print(f"🐍 [DEBUG] Project root: {project_root}")
print(f"🐍 [DEBUG] Current working directory: {os.getcwd()}")
print("🐍 [DEBUG] Python path:")
for i, path in enumerate(sys.path):
    print(f"🐍 [DEBUG]   {i}: {path}")

from PyQt5.QtWidgets import QApplication
from PyQt5.QtCore import QTimer, Qt
from PyQt5.QtGui import QFont, QPalette, QColor
from config.settings import APP_NAME
from config.theme import Theme
from core.ui.main_window import MainWindow
from utils.helpers import check_dependencies

def setup_logging():
    """Setup logging"""
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
        handlers=[logging.StreamHandler(sys.stdout)]
    )
    logger = logging.getLogger(APP_NAME)

    # Add debug handler for more verbose output
    debug_handler = logging.StreamHandler(sys.stdout)
    debug_handler.setLevel(logging.DEBUG)
    debug_formatter = logging.Formatter('🔍 [DEBUG] %(asctime)s - %(name)s - %(levelname)s - %(message)s')
    debug_handler.setFormatter(debug_formatter)
    logger.addHandler(debug_handler)
    logger.setLevel(logging.DEBUG)

    return logger

def check_module_imports():
    """Check if all required modules can be imported properly"""
    print("🔍 [DEBUG] Checking module imports...")

    modules_to_check = [
        'config.settings',
        'config.theme',
        'core.ui.main_window',
        'utils.helpers',
        'core.player',
        'core.ui.components.video_card',
        'core.ui.panels.search_panel',
        'core.ui.panels.playlist_panel'
    ]

    for module in modules_to_check:
        try:
            __import__(module)
            print(f"✅ [DEBUG] Successfully imported: {module}")
        except ImportError as e:
            print(f"❌ [DEBUG] Failed to import {module}: {e}")
            return False

    # Check for duplicate modules
    print("🔍 [DEBUG] Checking for duplicate modules...")
    import config.settings as settings1
    try:
        import ui.main_window as duplicate_ui
        print(f"⚠️  [DEBUG] WARNING: Found duplicate UI module at: {duplicate_ui.__file__}")
    except ImportError:
        print("✅ [DEBUG] No duplicate UI module found")

    return True

def main():
    """Main application entry point"""
    print("🚀 [DEBUG] =========================================")
    print("🚀 [DEBUG] Starting PurpleTube Player Main Function")
    print("🚀 [DEBUG] =========================================")

    logger = setup_logging()
    logger.info(f"Starting {APP_NAME}")

    # Check module imports first
    if not check_module_imports():
        logger.error("Module import check failed")
        return 1

    # Check dependencies
    print("🔍 [DEBUG] Checking dependencies...")
    if not check_dependencies():
        logger.error("Dependency check failed")
        return 1
    print("✅ [DEBUG] Dependency check passed")

    # Create application
    print("🔍 [DEBUG] Creating QApplication...")
    app = QApplication(sys.argv)
    app.setApplicationName(APP_NAME)
    print("✅ [DEBUG] QApplication created")

    # Setup font
    print("🔍 [DEBUG] Setting up application font...")
    font = QFont()
    font.setFamily("Inter")
    font.setPointSize(10)
    app.setFont(font)
    print("✅ [DEBUG] Font setup completed")

    # Setup palette
    print("🔍 [DEBUG] Setting up application palette...")
    palette = QPalette()
    palette.setColor(QPalette.Window, QColor(15, 15, 15))
    palette.setColor(QPalette.WindowText, Qt.white)
    palette.setColor(QPalette.Text, Qt.white)
    palette.setColor(QPalette.Highlight, QColor(106, 90, 205))
    app.setPalette(palette)
    print("✅ [DEBUG] Palette setup completed")

    # Create main window
    print("🔍 [DEBUG] Creating MainWindow...")
    try:
        window = MainWindow()
        print(f"✅ [DEBUG] MainWindow created successfully: {window}")
        print(f"🔍 [DEBUG] MainWindow class: {window.__class__}")
        print(f"🔍 [DEBUG] MainWindow module: {window.__class__.__module__}")

        # Verify queue functionality is properly connected
        print("🔍 [DEBUG] Verifying queue connections...")
        if hasattr(window, 'vlc_player') and hasattr(window.vlc_player, 'queue_manager'):
            print("✅ [DEBUG] VLC player and queue manager found in MainWindow")

            # Verify search panel connections
            if hasattr(window, 'search_panel') and hasattr(window.search_panel, 'add_to_queue_requested'):
                print("✅ [DEBUG] Search panel queue connections verified")

            # Verify playlist panel connections
            if hasattr(window, 'playlist_panel') and hasattr(window.playlist_panel, 'add_to_queue_requested'):
                print("✅ [DEBUG] Playlist panel queue connections verified")

            # Verify feed content panel connections
            if hasattr(window, 'feed_content_panel') and hasattr(window.feed_content_panel, 'add_to_queue_requested'):
                print("✅ [DEBUG] Feed content panel queue connections verified")

            # Verify video card connections
            if hasattr(window, 'search_panel') and hasattr(window.search_panel, 'video_grid_layout'):
                print("✅ [DEBUG] Video card layout found in search panel")

    except Exception as e:
        print(f"❌ [DEBUG] Failed to create MainWindow: {e}")
        import traceback
        traceback.print_exc()
        return 1

    # Show window
    print("🔍 [DEBUG] Showing MainWindow...")
    window.show()
    print("✅ [DEBUG] MainWindow shown")

    # Setup cleanup timer for debug info
    debug_timer = QTimer()
    debug_timer.timeout.connect(lambda: None)  # Just to have the timer
    debug_timer.start(100)  # Small timer to keep things running

    def print_final_debug():
        print("✅ [DEBUG] =========================================")
        print("✅ [DEBUG] Application started successfully!")
        print("✅ [DEBUG] =========================================")
        print("✅ [DEBUG] MainWindow properties:")
        print(f"✅ [DEBUG]   - Visible: {window.isVisible()}")
        print(f"✅ [DEBUG]   - Geometry: {window.geometry()}")
        print(f"✅ [DEBUG]   - Search input exists: {hasattr(window, 'search_input')}")
        if hasattr(window, 'search_input'):
            print(f"✅ [DEBUG]   - Search input type: {type(window.search_input)}")
        print(f"✅ [DEBUG]   - Search panel exists: {hasattr(window, 'search_panel')}")
        if hasattr(window, 'search_panel'):
            print(f"✅ [DEBUG]   - Search panel type: {type(window.search_panel)}")
            print(f"✅ [DEBUG]   - Search panel queue signal: {hasattr(window.search_panel, 'add_to_queue_requested')}")
        print(f"✅ [DEBUG]   - Playlist panel exists: {hasattr(window, 'playlist_panel')}")
        if hasattr(window, 'playlist_panel'):
            print(f"✅ [DEBUG]   - Playlist panel type: {type(window.playlist_panel)}")
            print(f"✅ [DEBUG]   - Playlist panel queue signal: {hasattr(window.playlist_panel, 'add_to_queue_requested')}")
        print(f"✅ [DEBUG]   - VLC player exists: {hasattr(window, 'vlc_player')}")
        if hasattr(window, 'vlc_player'):
            print(f"✅ [DEBUG]   - VLC player type: {type(window.vlc_player)}")
            print(f"✅ [DEBUG]   - Queue manager exists: {hasattr(window.vlc_player, 'queue_manager')}")
        print("✅ [DEBUG] =========================================")

    # Print final debug info after event loop starts
    QTimer.singleShot(500, print_final_debug)

    # Start application
    logger.info("Application started successfully")
    print("🔍 [DEBUG] Entering application event loop...")

    try:
        result = app.exec_()
        print(f"🔍 [DEBUG] Application exited with code: {result}")
        return result
    except Exception as e:
        print(f"❌ [DEBUG] Exception in event loop: {e}")
        import traceback
        traceback.print_exc()
        return 1

if __name__ == "__main__":
    print("🔍 [DEBUG] =========================================")
    print("🔍 [DEBUG] PurpleTube Player Starting")
    print("🔍 [DEBUG] =========================================")

    try:
        exit_code = main()
        print(f"🔍 [DEBUG] Application finished with exit code: {exit_code}")
        sys.exit(exit_code)
    except Exception as e:
        logging.critical(f"Application crashed: {e}")
        print(f"💥 [DEBUG] CRITICAL ERROR: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)

