from PyQt5.QtWidgets import (QWidget, QVBoxLayout, QPushButton, QLabel, 
                            QSpacerItem, QSizePolicy, QMessageBox, QGraphicsOpacityEffect)
from PyQt5.QtCore import pyqtSignal, Qt, QPropertyAnimation, QEasingCurve, QTimer, QParallelAnimationGroup, QSequentialAnimationGroup, pyqtProperty
from PyQt5.QtGui import QFont, QFontDatabase, QPainter, QColor, QLinearGradient, QPalette
from typing import Dict, List, Tuple
from dataclasses import dataclass
from config.theme import Theme
import random
import math


@dataclass
class NavigationItem:
    """Data class for navigation items"""
    text: str
    view: str
    icon: str


class AnimatedButton(QPushButton):
    """Custom button class with animation properties"""
    
    def __init__(self, text, parent=None):
        super().__init__(text, parent)
        self._opacity = 1.0
        self._scale = 1.0
        self._rotation = 0
        self._hue_rotate = 0
        
        # Setup graphics effects
        self.opacity_effect = QGraphicsOpacityEffect(self)
        self.setGraphicsEffect(self.opacity_effect)
    
    @pyqtProperty(float)
    def scale(self):
        return self._scale
    
    @scale.setter
    def scale(self, value):
        self._scale = value
        self.update()
    
    @pyqtProperty(float)
    def rotation(self):
        return self._rotation
    
    @rotation.setter
    def rotation(self, value):
        self._rotation = value
        self.update()
    
    @pyqtProperty(float)
    def hue_rotate(self):
        return self._hue_rotate
    
    @hue_rotate.setter
    def hue_rotate(self, value):
        self._hue_rotate = value
        self.update()


class Sidebar(QWidget):
    """Modern sidebar navigation with enhanced animations and better architecture"""
    
    view_changed = pyqtSignal(str)
    
    # Navigation configuration
    NAV_ITEMS: Tuple[NavigationItem] = (
        NavigationItem("Home", "home", "🔍"),
        NavigationItem("Playlist", "playlists", "🎵"),
        NavigationItem("Iscrizioni", "subscriptions", "📡"),
        NavigationItem("Player", "vlc", "🎬"),
        NavigationItem("Impostazioni", "settings", "⚙️")
    )

    def __init__(self, main_window):
        super().__init__()
        self.main_window = main_window
        self.button_map: Dict[str, AnimatedButton] = {}
        self.click_counters: Dict[str, int] = {}
        self.current_view: str = ""
        self.animation = None
        self.easter_egg_active = False
        self.active_animations = []
        self.setup_ui()

    def setup_ui(self):
        """Initialize the modern UI"""
        self.setFixedWidth(240)
        self.apply_styles()
        
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)

        self.setup_title(layout)
        self.setup_navigation(layout)
        layout.addStretch()  # Push content to top
        self.setup_footer(layout)

    def apply_styles(self):
        """Apply modern CSS styles"""
        self.setStyleSheet(f"""
            Sidebar {{
                background: {Theme.BACKGROUND_LIGHT};
                border-right: {Theme.BORDER_WIDTH} solid {Theme.BORDER};
                font-family: {Theme.FONT_FAMILY};
            }}
            
            QPushButton {{
                border: none;
                text-align: left;
                padding: 12px 20px;
                font-size: 20px;
                font-family: {Theme.FONT_FAMILY};
                background: transparent;
                color: {Theme.TEXT_SECONDARY};
            }}
            
            QPushButton:hover {{
                background: {Theme.SURFACE_HOVER};
                color: {Theme.TEXT_PRIMARY};
                padding-left: 22px;  /* Slight shift on hover */
            }}
            
            QPushButton[active="true"] {{
                background: {Theme.SURFACE_ACTIVE};
                color: {Theme.TEXT_PRIMARY};
                font-weight: bold;
                border-left: 3px solid {Theme.PRIMARY};
            }}
            
            QLabel#sidebar-title {{
                font-size: 18px;
                font-weight: bold;
                color: {Theme.TEXT_PRIMARY};
                padding: 20px;
                background: {Theme.BACKGROUND_LIGHT};
                border-bottom: 1px solid {Theme.BORDER};
            }}
        """)

    def setup_title(self, layout: QVBoxLayout):
        """Setup the sidebar title with modern styling"""
        title = QLabel("PurpleTube Player")
        title.setObjectName("sidebar-title")
        title.setAlignment(Qt.AlignCenter)
        layout.addWidget(title)

    def setup_navigation(self, layout: QVBoxLayout):
        """Setup navigation buttons with modern approach"""
        for nav_item in self.NAV_ITEMS:
            btn = self.create_nav_button(nav_item)
            layout.addWidget(btn)
            self.button_map[nav_item.view] = btn
            self.click_counters[nav_item.view] = 0

    def create_nav_button(self, nav_item: NavigationItem) -> AnimatedButton:
        """Create a modern navigation button"""
        btn = AnimatedButton(f"{nav_item.icon}  {nav_item.text}")
        btn.setProperty("view", nav_item.view)
        btn.setCursor(Qt.PointingHandCursor)
        
        # Use lambda with captured variable to avoid late binding issues
        btn.clicked.connect(lambda checked, v=nav_item.view: self.on_button_clicked(v))
        
        return btn

    def setup_footer(self, layout: QVBoxLayout):
        """Optional: Add footer section"""
        pass

    def on_button_clicked(self, view: str):
        """Handle button clicks with animation and easter eggs"""
        # Increment click counter
        self.click_counters[view] += 1
        
        # Check for easter egg trigger only for settings button
        if view == "settings" and self.click_counters[view] >= 10 and not self.easter_egg_active:
            self.trigger_easter_egg()
            self.click_counters[view] = 0  # Reset counter after triggering
        
        self.animate_button_click(view)
        self.view_changed.emit(view)
        self.set_active_view(view)

    def trigger_easter_egg(self):
        """Trigger matrix-style easter egg for settings button"""
        self.easter_egg_active = True
        
        tech_emojis = ["⚙️", "🔧", "🛠️", "💻", "📱", "🔌", "💾", "🖥️"]
        matrix_colors = ["#00FF00", "#00CC00", "#009900", "#00FF66"]
        
        # Digital rain effect
        for i, btn in enumerate(self.button_map.values()):
            color = matrix_colors[i % len(matrix_colors)]
            QTimer.singleShot(i * 80, lambda b=btn, c=color: self.animate_matrix_effect(b, tech_emojis, c))
        
        # Binary code background
        self.animate_binary_rain()
        
        # Glitch effect
        self.animate_glitch_effect()
        
        self.show_easter_egg_message("⚙️ Sistema Hackerato!", "Sei entrato nella matrix di PurpleTube! 💻")
        
        # Reset easter egg flag after a delay
        QTimer.singleShot(5000, lambda: setattr(self, 'easter_egg_active', False))

    def animate_matrix_effect(self, button, emojis, color):
        """Matrix-style digital rain effect"""
        original_text = button.text()
        
        # Multiple glitch animations
        for i in range(5):
            QTimer.singleShot(i * 200, lambda: self.matrix_glitch_frame(button, emojis, color, original_text))
        
        # Final restore
        QTimer.singleShot(1200, lambda: button.setText(original_text))

    def matrix_glitch_frame(self, button, emojis, color, original_text):
        """Single frame of matrix glitch effect"""
        glitch_text = f"{random.choice(emojis)} {''.join(random.choice('01') for _ in range(8))}"
        button.setText(glitch_text)
        button.setStyleSheet(f"color: {color}; font-family: 'Courier New';")

    def animate_binary_rain(self):
        """Animate binary code rain in background"""
        for _ in range(10):
            binary_label = QLabel("".join(random.choice("01") for _ in range(8)), self)
            binary_label.setStyleSheet("color: #00FF00; font-family: 'Courier New'; font-size: 10px;")
            binary_label.move(random.randint(0, self.width()), -20)
            binary_label.show()
            
            # Fall animation
            anim = QPropertyAnimation(binary_label, b"pos")
            anim.setDuration(2000)
            anim.setStartValue(binary_label.pos())
            anim.setEndValue(binary_label.pos().translated(0, self.height() + 20))
            anim.start()
            
            QTimer.singleShot(2000, binary_label.deleteLater)

    def animate_glitch_effect(self):
        """Animate digital glitch effect on entire sidebar"""
        original_geometry = self.geometry()
        
        for i in range(10):
            QTimer.singleShot(i * 100, lambda: self.single_glitch_frame(original_geometry))
        
        QTimer.singleShot(1200, lambda: self.move(original_geometry.topLeft()))

    def single_glitch_frame(self, original_geometry):
        """Single frame of glitch effect"""
        offset_x = random.randint(-5, 5)
        offset_y = random.randint(-2, 2)
        self.move(original_geometry.topLeft().x() + offset_x, original_geometry.topLeft().y() + offset_y)

    def show_easter_egg_message(self, title, message):
        """Show enhanced easter egg discovery message"""
        msg = QMessageBox()
        msg.setWindowTitle(f"🎉 {title}")
        msg.setText(message)
        msg.setIcon(QMessageBox.Information)
        msg.setStandardButtons(QMessageBox.Ok)
        
        # Enhanced styling
        msg.setStyleSheet(f"""
            QMessageBox {{
                background: qlineargradient(x1:0, y1:0, x2:1, y2:1,
                                         stop:0 #667eea, stop:1 #764ba2);
                color: white;
                font-weight: bold;
            }}
            QMessageBox QLabel {{
                color: white;
                font-size: 14px;
                background: transparent;
            }}
            QPushButton {{
                background: qlineargradient(x1:0, y1:0, x2:1, y2:1,
                                         stop:0 #FF6B6B, stop:1 #FFE66D);
                color: black;
                padding: 8px 16px;
                border-radius: 8px;
                font-weight: bold;
                border: 2px solid white;
            }}
            QPushButton:hover {{
                background: qlineargradient(x1:0, y1:0, x2:1, y2:1,
                                         stop:0 #FF8E53, stop:1 #FFE66D);
            }}
        """)
        
        msg.exec_()

    def animate_button_click(self, view: str):
        """Enhanced button click animation"""
        btn = self.button_map[view]
        
        group = QParallelAnimationGroup()
        
        # Scale animation
        scale_anim = QPropertyAnimation(btn, b"scale")
        scale_anim.setDuration(200)
        scale_anim.setKeyValueAt(0, 1.0)
        scale_anim.setKeyValueAt(0.5, 0.95)
        scale_anim.setKeyValueAt(1, 1.0)
        group.addAnimation(scale_anim)
        
        # Opacity animation
        opacity_anim = QPropertyAnimation(btn.opacity_effect, b"opacity")
        opacity_anim.setDuration(200)
        opacity_anim.setKeyValueAt(0, 1.0)
        opacity_anim.setKeyValueAt(0.5, 0.7)
        opacity_anim.setKeyValueAt(1, 1.0)
        group.addAnimation(opacity_anim)
        
        group.start()

    def set_active_view(self, view: str):
        """Set the active view with proper state management"""
        if view == self.current_view:
            return
            
        self.current_view = view
        
        for btn_view, btn in self.button_map.items():
            is_active = btn_view == view
            btn.setProperty("active", is_active)
            
            # Force style update
            btn.style().unpolish(btn)
            btn.style().polish(btn)
            btn.update()

    def update_theme(self):
        """Update styles when theme changes"""
        self.apply_styles()
        for btn in self.button_map.values():
            btn.style().unpolish(btn)
            btn.style().polish(btn)

    def resizeEvent(self, event):
        """Handle resize events if needed"""
        super().resizeEvent(event)

    @property
    def available_views(self) -> List[str]:
        """Get list of available views"""
        return [item.view for item in self.NAV_ITEMS]

    def reset_click_counters(self):
        """Reset all click counters (useful for testing)"""
        for view in self.click_counters:
            self.click_counters[view] = 0
