"""
RSS Feed Service for PurpleTube Player.
This module handles all RSS feed-related operations including parsing, adding,
refreshing, and retrieving feeds and feed items from the database.
"""
import logging
import threading
import time
from datetime import datetime, timedelta
from typing import List, Optional, Tuple, Dict, Any
from urllib.parse import urlparse
import feedparser
import requests
from tenacity import retry, stop_after_attempt, wait_exponential
from functools import lru_cache
from bs4 import BeautifulSoup
from config.logging_config import logger
from config.settings import API_RATE_LIMITER
from core.models import RSSFeed, FeedItem, Video
from core.database import db_manager

class RSSService:
    """
    Service class for managing RSS feeds.
    Provides methods for parsing, adding, refreshing, and retrieving RSS feeds
    and their items from the database with pagination support.
    """
    SESSION = requests.Session()
    SESSION.headers.update({"User-Agent": "purpleTubePlayer/1.0"})
    _lock = threading.Lock()

    @staticmethod
    def validate_url(url: str) -> bool:
        """
        Validate a URL.
        Args:
            url: The URL to validate.
        Returns:
            bool: True if the URL is valid.
        Raises:
            ValueError: If the URL is invalid.
        """
        if not url.startswith(("http://", "https://")):
            raise ValueError("Invalid URL: Must start with http:// or https://")
        parsed_url = urlparse(url)
        if not all([parsed_url.scheme, parsed_url.netloc]):
            raise ValueError("Invalid URL: Malformed or missing components.")
        return True

    @staticmethod
    def secure_request(url: str, **kwargs) -> requests.Response:
        """
        Make a secure HTTP request.
        Args:
            url: The URL to request.
            **kwargs: Additional arguments to pass to requests.get().
        Returns:
            requests.Response: The response object.
        Raises:
            RuntimeError: If the API rate limit is exceeded.
            ValueError: If the URL is invalid.
        """
        RSSService.validate_url(url)
        if not API_RATE_LIMITER.increment():
            raise RuntimeError("API call limit exceeded. Please try again later.")
        return requests.get(url, verify=True, **kwargs)

    @staticmethod
    @lru_cache(maxsize=32)
    def get_feed_items_cached(feed_id: int, limit: int = None, offset: int = 0) -> List[FeedItem]:
        """
        Get cached feed items for a specific feed with pagination.
        Args:
            feed_id: The ID of the feed.
            limit: Optional limit on the number of items to return.
            offset: Offset for pagination.
        Returns:
            List[FeedItem]: A list of feed items.
        """
        return RSSService.get_feed_items(feed_id, limit, offset)

    @staticmethod
    @retry(stop=stop_after_attempt(3), wait=wait_exponential(multiplier=1, min=2, max=10))
    def parse_peertube_feed(feed_url: str) -> Tuple[Optional[RSSFeed], List[FeedItem]]:
        """
        Parse a PeerTube RSS feed.
        Args:
            feed_url: The URL of the RSS feed.
        Returns:
            Tuple[Optional[RSSFeed], List[FeedItem]]: A tuple containing the feed and its items.
        """
        try:
            logger.info(f"Parsing PeerTube feed: {feed_url}")
            if not API_RATE_LIMITER.increment():
                raise RuntimeError("API call limit exceeded. Please try again later.")
            RSSService.validate_url(feed_url)
            # Fetch and parse the feed
            logger.info("Fetching feed with feedparser...")
            response = RSSService.secure_request(feed_url)
            feed = feedparser.parse(response.text)
            logger.info(f"Feed entries found: {len(feed.entries)}")
            if not feed.entries:
                logger.warning(f"Empty or invalid feed: {feed_url}")
                return None, []
            # Determine if this is a channel feed
            is_channel = "account" in feed.feed.get("link", "").lower() if feed.feed else False
            rss_feed = RSSFeed(
                url=feed_url,
                title=feed.feed.get("title", "Untitled Feed"),
                last_updated=datetime.now(),
                is_channel=is_channel
            )
            # Parse feed items
            items = []
            for entry in feed.entries:
                try:
                    logger.debug(f"Processing entry: {entry.get('title', 'Untitled')}")
                    # Extract duration
                    duration = "0"
                    for tag in entry.get("media_content", []):
                        if isinstance(tag, dict) and tag.get("medium") == "video":
                            duration = str(tag.get("duration", "0"))
                            break
                    # Extract thumbnail
                    thumbnail_url = ""
                    if hasattr(entry, "media_thumbnail"):
                        thumbnail_url = entry.media_thumbnail[0]["url"] if entry.media_thumbnail else ""
                    elif hasattr(entry, "enclosures") and entry.enclosures:
                        thumbnail_url = entry.enclosures[0].get("url", "")
                    # Extract video URL and ID
                    video_url = entry.link
                    video_id = ""
                    if video_url:
                        parsed = urlparse(video_url)
                        if parsed.path:
                            video_id = parsed.path.split("/")[-1]
                    # Extract publish date
                    published = datetime.now()
                    if hasattr(entry, "published_parsed"):
                        published = datetime(*entry.published_parsed[:6])
                    # Create FeedItem
                    items.append(FeedItem(
                        title=entry.get("title", "No Title"),
                        url=video_url,
                        thumbnail_url=thumbnail_url,
                        duration=duration,
                        video_id=video_id,
                        published=published,
                        author=entry.get("author", "Unknown"),
                        platform="PeerTube"
                    ))
                except Exception as e:
                    logger.error(f"Error parsing feed item: {e}", exc_info=True)
            logger.info(f"Successfully parsed {len(items)} items from feed.")
            return rss_feed, items
        except Exception as e:
            logger.error(f"Error parsing RSS feed: {e}", exc_info=True)
            return None, []

    @staticmethod
    def add_feed(feed_url: str) -> bool:
        """
        Add a new RSS feed to the database.
        Args:
            feed_url: The URL of the RSS feed.
        Returns:
            bool: True if the feed was added successfully.
        """
        try:
            logger.info(f"Adding RSS feed: {feed_url}")
            feed, items = RSSService.parse_peertube_feed(feed_url)
            if not feed:
                logger.error("Failed to parse RSS feed.")
                return False
            with RSSService._lock:
                with db_manager.get_connection() as conn:
                    cursor = conn.cursor()
                    cursor.execute("""
                        INSERT INTO rss_feeds (url, title, last_updated, is_channel)
                        VALUES (?, ?, ?, ?)
                    """, (feed.url, feed.title, feed.last_updated.isoformat(), int(feed.is_channel)))
                    feed_id = cursor.lastrowid
                    cursor.executemany("""
                        INSERT OR IGNORE INTO feed_items
                        (feed_id, video_id, title, url, thumbnail_url, duration, published, author, platform)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    """, [
                        (
                            feed_id, item.video_id, item.title, item.url, item.thumbnail_url,
                            item.duration, item.published.isoformat(), item.author, item.platform
                        )
                        for item in items
                    ])
                    conn.commit()
            logger.info(f"Successfully added RSS feed: {feed.title}")
            return True
        except Exception as e:
            logger.error(f"Error adding feed: {e}", exc_info=True)
            return False

    @staticmethod
    def get_feeds() -> List[RSSFeed]:
        """
        Get all RSS feeds from the database.
        Returns:
            List[RSSFeed]: A list of all RSS feeds.
        """
        with db_manager.get_connection() as conn:
            cursor = conn.cursor()
            cursor.execute("SELECT id, url, title, last_updated, is_channel FROM rss_feeds")
            return [
                RSSFeed(
                    id=row[0],
                    url=row[1],
                    title=row[2],
                    last_updated=datetime.fromisoformat(row[3]) if row[3] else None,
                    is_channel=bool(row[4])
                )
                for row in cursor.fetchall()
            ]

    @staticmethod
    def get_feed_items(
        feed_id: int,
        limit: int = None,
        offset: int = 0,
        published_after: datetime = None
    ) -> List[FeedItem]:
        """
        Get feed items for a specific feed with pagination.
        Args:
            feed_id: The ID of the feed.
            limit: Optional limit on the number of items to return.
            offset: Offset for pagination.
            published_after: Optional datetime to filter items published after.
        Returns:
            List[FeedItem]: A list of feed items.
        """
        with db_manager.get_connection() as conn:
            cursor = conn.cursor()
            query = """
                SELECT video_id, title, url, thumbnail_url, duration, published, author, platform
                FROM feed_items
                WHERE feed_id = ?
            """
            params = [feed_id]
            if published_after:
                query += " AND published >= ?"
                params.append(published_after.isoformat())
            query += " ORDER BY published DESC"
            if limit is not None:
                query += " LIMIT ? OFFSET ?"
                params.extend([limit, offset])
            else:
                query += " OFFSET ?"
                params.append(offset)
            cursor.execute(query, params)
            return [
                FeedItem(
                    video_id=row[0],
                    title=row[1],
                    url=row[2],
                    thumbnail_url=row[3],
                    duration=row[4],
                    published=datetime.fromisoformat(row[5]),
                    author=row[6],
                    platform=row[7]
                )
                for row in cursor.fetchall()
            ]

    @staticmethod
    def get_feed_videos(
        feed_id: int,
        limit: int = None,
        offset: int = 0
    ) -> List[Video]:
        """
        Get feed items as Video objects for a specific feed with pagination.
        Args:
            feed_id: The ID of the feed.
            limit: Optional limit on the number of items to return.
            offset: Offset for pagination.
        Returns:
            List[Video]: A list of Video objects.
        """
        feed_items = RSSService.get_feed_items(feed_id, limit, offset)
        return [
            Video(
                title=item.title,
                url=item.url,
                thumbnail_url=item.thumbnail_url,
                duration=item.duration,
                video_id=item.video_id,
                platform=item.platform,
                author=item.author,
                published=item.published
            )
            for item in feed_items
        ]

    @staticmethod
    def remove_feed(feed_id: int) -> bool:
        """
        Remove an RSS feed from the database.
        Args:
            feed_id: The ID of the feed to remove.
        Returns:
            bool: True if the feed was removed successfully.
        """
        try:
            with RSSService._lock:
                with db_manager.get_connection() as conn:
                    cursor = conn.cursor()
                    cursor.execute("DELETE FROM rss_feeds WHERE id = ?", (feed_id,))
                    conn.commit()
            return True
        except Exception as e:
            logger.error(f"Error removing feed: {e}")
            return False

    @staticmethod
    def refresh_feed(feed_id: int) -> bool:
        """
        Refresh a specific RSS feed.
        Args:
            feed_id: The ID of the feed to refresh.
        Returns:
            bool: True if the feed was refreshed successfully.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT url FROM rss_feeds WHERE id = ?", (feed_id,))
                row = cursor.fetchone()
                if not row:
                    return False
                feed_url = row[0]
            feed, items = RSSService.parse_peertube_feed(feed_url)
            if not feed:
                return False
            with RSSService._lock:
                with db_manager.get_connection() as conn:
                    cursor = conn.cursor()
                    cursor.execute("""
                        UPDATE rss_feeds
                        SET title = ?, last_updated = ?
                        WHERE id = ?
                    """, (feed.title, datetime.now().isoformat(), feed_id))
                    existing_ids = {row[0] for row in cursor.execute(
                        "SELECT video_id FROM feed_items WHERE feed_id = ?", (feed_id,)
                    ).fetchall()}
                    new_items = [
                        (
                            feed_id, item.video_id, item.title, item.url, item.thumbnail_url,
                            item.duration, item.published.isoformat(), item.author, item.platform
                        )
                        for item in items if item.video_id not in existing_ids
                    ]
                    if new_items:
                        cursor.executemany("""
                            INSERT INTO feed_items
                            (feed_id, video_id, title, url, thumbnail_url, duration, published, author, platform)
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                        """, new_items)
                    conn.commit()
            return True
        except Exception as e:
            logger.error(f"Error refreshing feed {feed_id}: {e}")
            return False

    @staticmethod
    def refresh_all_feeds() -> bool:
        """
        Refresh all RSS feeds.
        Returns:
            bool: True if all feeds were refreshed successfully.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT id, url FROM rss_feeds")
                feeds = cursor.fetchall()
            def refresh_single_feed(feed_id, feed_url):
                try:
                    feed, items = RSSService.parse_peertube_feed(feed_url)
                    if not feed:
                        return False
                    with RSSService._lock:
                        with db_manager.get_connection() as conn:
                            cursor = conn.cursor()
                            cursor.execute("""
                                UPDATE rss_feeds
                                SET title = ?, last_updated = ?
                                WHERE id = ?
                            """, (feed.title, datetime.now().isoformat(), feed_id))
                            existing_ids = {row[0] for row in cursor.execute(
                                "SELECT video_id FROM feed_items WHERE feed_id = ?", (feed_id,)
                            ).fetchall()}
                            new_items = [
                                (
                                    feed_id, item.video_id, item.title, item.url, item.thumbnail_url,
                                    item.duration, item.published.isoformat(), item.author, item.platform
                                )
                                for item in items if item.video_id not in existing_ids
                            ]
                            if new_items:
                                cursor.executemany("""
                                    INSERT INTO feed_items
                                    (feed_id, video_id, title, url, thumbnail_url, duration, published, author, platform)
                                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                                """, new_items)
                            conn.commit()
                    return True
                except Exception as e:
                    logger.error(f"Error refreshing feed {feed_id}: {e}")
                    return False
            from concurrent.futures import ThreadPoolExecutor, as_completed
            with ThreadPoolExecutor(max_workers=4) as executor:
                futures = [executor.submit(refresh_single_feed, feed_id, feed_url) for feed_id, feed_url in feeds]
                results = [future.result() for future in as_completed(futures)]
            return all(results)
        except Exception as e:
            logger.error(f"Error refreshing feeds: {e}")
            return False

    @staticmethod
    def get_all_feed_items(
        limit: int = None,
        offset: int = 0,
        published_after: datetime = None
    ) -> List[FeedItem]:
        """
        Get all feed items from all feeds with pagination.
        Args:
            limit: Optional limit on the number of items to return.
            offset: Offset for pagination.
            published_after: Optional datetime to filter items published after.
        Returns:
            List[FeedItem]: A list of all feed items.
        """
        with db_manager.get_connection() as conn:
            cursor = conn.cursor()
            query = """
                SELECT fi.video_id, fi.title, fi.url, fi.thumbnail_url, fi.duration,
                       fi.published, fi.author, fi.platform
                FROM feed_items fi
                JOIN rss_feeds f ON fi.feed_id = f.id
            """
            params = []
            if published_after:
                query += " WHERE fi.published >= ?"
                params.append(published_after.isoformat())
            query += " ORDER BY fi.published DESC"
            if limit is not None:
                query += " LIMIT ? OFFSET ?"
                params.extend([limit, offset])
            else:
                query += " OFFSET ?"
                params.append(offset)
            cursor.execute(query, params)
            return [
                FeedItem(
                    video_id=row[0],
                    title=row[1],
                    url=row[2],
                    thumbnail_url=row[3],
                    duration=row[4],
                    published=datetime.fromisoformat(row[5]),
                    author=row[6],
                    platform=row[7]
                )
                for row in cursor.fetchall()
            ]

    @staticmethod
    def get_all_feed_items_as_videos(
        limit: int = None,
        offset: int = 0,
        published_after: datetime = None
    ) -> List[Video]:
        """
        Get all feed items as Video objects from all feeds with pagination.
        Args:
            limit: Optional limit on the number of items to return.
            offset: Offset for pagination.
            published_after: Optional datetime to filter items published after.
        Returns:
            List[Video]: A list of Video objects.
        """
        feed_items = RSSService.get_all_feed_items(limit, offset, published_after)
        return [
            Video(
                title=item.title,
                url=item.url,
                thumbnail_url=item.thumbnail_url,
                duration=item.duration,
                video_id=item.video_id,
                platform=item.platform,
                author=item.author,
                published=item.published
            )
            for item in feed_items
        ]

    @staticmethod
    def get_all_feed_items_with_timeout(
        limit: int = None,
        offset: int = 0,
        timeout: int = 30
    ) -> List[FeedItem]:
        """
        Get all feed items with a timeout.
        Args:
            limit: Optional limit on the number of items to return.
            offset: Offset for pagination.
            timeout: Timeout in seconds.
        Returns:
            List[FeedItem]: A list of all feed items.
        """
        def worker():
            try:
                return RSSService.get_all_feed_items(limit, offset)
            except Exception as e:
                logger.error(f"Error in worker thread: {e}")
                return []
        from concurrent.futures import ThreadPoolExecutor
        with ThreadPoolExecutor(max_workers=1) as executor:
            future = executor.submit(worker)
            try:
                return future.result(timeout=timeout)
            except TimeoutError:
                logger.error("Timeout while fetching feed items")
                future.cancel()
                return []
            except Exception as e:
                logger.error(f"Error fetching feed items: {e}")
                return []

    @staticmethod
    def clear_all_feeds() -> bool:
        """
        Clear all feeds from the database.
        Returns:
            bool: True if all feeds were cleared successfully.
        """
        try:
            with RSSService._lock:
                with db_manager.get_connection() as conn:
                    cursor = conn.cursor()
                    cursor.execute("DELETE FROM feed_items")
                    cursor.execute("DELETE FROM rss_feeds")
                    conn.commit()
            return True
        except Exception as e:
            logger.error(f"Error clearing feeds: {e}")
            return False

    @staticmethod
    def get_feed_items_as_videos(
        feed_id: int,
        limit: int = None,
        offset: int = 0
    ) -> List[Video]:
        """
        Get feed items as Video objects for a specific feed with pagination.
        Args:
            feed_id: The ID of the feed.
            limit: Optional limit on the number of items to return.
            offset: Offset for pagination.
        Returns:
            List[Video]: A list of Video objects.
        """
        return RSSService.get_feed_videos(feed_id, limit, offset)

