# core/database/__init__.py
import os
import sqlite3
from pathlib import Path
from datetime import datetime
from typing import Optional, List, Dict, Tuple
from config.settings import DATA_DIR, DB_PATH, DBConfig
from config.logging_config import logger

class DBManager:
    """Database manager for PurpleTube Player."""

    def __init__(self) -> None:
        """Initialize the database manager."""
        self.DB_PATH = DB_PATH
        self._init_db()
        self._create_indices()
        self._verify_db_integrity()

    def _init_db(self) -> None:
        """Initialize the database tables."""
        try:
            with sqlite3.connect(self.DB_PATH) as conn:
                conn.execute("PRAGMA foreign_keys = ON")
                for key, value in DBConfig.PRAGMA.items():
                    conn.execute(f"PRAGMA {key} = {value}")

                conn.execute("""
                CREATE TABLE IF NOT EXISTS playlists (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    name TEXT UNIQUE NOT NULL,
                    created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                    updated_at TEXT DEFAULT CURRENT_TIMESTAMP
                );
                """)
                conn.execute("""
                CREATE TABLE IF NOT EXISTS playlist_videos (
                    playlist_id INTEGER NOT NULL,
                    video_id TEXT NOT NULL,
                    position INTEGER DEFAULT 0,
                    title TEXT NOT NULL,
                    url TEXT NOT NULL,
                    thumbnail_url TEXT,
                    duration TEXT DEFAULT '0',
                    platform TEXT DEFAULT 'PeerTube',
                    author TEXT DEFAULT 'Unknown',
                    published TEXT,
                    added_at TEXT DEFAULT CURRENT_TIMESTAMP,
                    PRIMARY KEY (playlist_id, video_id),
                    FOREIGN KEY (playlist_id) REFERENCES playlists(id) ON DELETE CASCADE
                );
                """)
                conn.execute("""
                CREATE TABLE IF NOT EXISTS rss_feeds (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    url TEXT UNIQUE NOT NULL,
                    title TEXT NOT NULL,
                    last_updated TEXT,
                    is_channel INTEGER DEFAULT 0,
                    created_at TEXT DEFAULT CURRENT_TIMESTAMP
                );
                """)
                conn.execute("""
                CREATE TABLE IF NOT EXISTS feed_items (
                    feed_id INTEGER NOT NULL,
                    video_id TEXT NOT NULL,
                    title TEXT NOT NULL,
                    url TEXT NOT NULL,
                    thumbnail_url TEXT,
                    duration TEXT DEFAULT '0',
                    published TEXT NOT NULL,
                    author TEXT DEFAULT 'Unknown',
                    platform TEXT DEFAULT 'PeerTube',
                    added_at TEXT DEFAULT CURRENT_TIMESTAMP,
                    PRIMARY KEY (feed_id, video_id),
                    FOREIGN KEY (feed_id) REFERENCES rss_feeds(id) ON DELETE CASCADE
                );
                """)
                conn.execute("""
                CREATE TABLE IF NOT EXISTS user_preferences (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    key TEXT UNIQUE NOT NULL,
                    value TEXT,
                    updated_at TEXT DEFAULT CURRENT_TIMESTAMP
                );
                """)
                conn.execute("INSERT OR IGNORE INTO playlists (name) VALUES ('Watch Later')")
                conn.execute("INSERT OR IGNORE INTO user_preferences (key, value) VALUES ('player', 'Default Browser')")
                conn.commit()
                logger.info(f"Database initialized at: {self.DB_PATH}")
        except sqlite3.Error as e:
            logger.error(f"Error initializing database: {e}")
            raise

    def _create_indices(self) -> None:
        """Create database indices for performance."""
        with sqlite3.connect(self.DB_PATH) as conn:
            conn.execute("CREATE INDEX IF NOT EXISTS idx_playlist_videos_playlist ON playlist_videos(playlist_id)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_playlist_videos_position ON playlist_videos(playlist_id, position)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_playlist_videos_video ON playlist_videos(video_id)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_playlist_videos_added ON playlist_videos(added_at)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_feed_items_feed ON feed_items(feed_id)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_feed_items_published ON feed_items(published)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_feed_items_video ON feed_items(video_id)")
            conn.execute("CREATE INDEX IF NOT EXISTS idx_feed_items_added ON feed_items(added_at)")
            conn.commit()

    def _verify_db_integrity(self) -> None:
        """Verify and repair database integrity."""
        with sqlite3.connect(self.DB_PATH) as conn:
            cursor = conn.cursor()
            cursor.execute("SELECT 1 FROM playlists WHERE name = 'Watch Later'")
            if not cursor.fetchone():
                cursor.execute("INSERT INTO playlists (name) VALUES ('Watch Later')")
            cursor.execute("""
            DELETE FROM playlist_videos
            WHERE playlist_id NOT IN (SELECT id FROM playlists)
            """)
            cursor.execute("""
            DELETE FROM feed_items
            WHERE feed_id NOT IN (SELECT id FROM rss_feeds)
            """)
            conn.commit()

    def get_connection(self) -> sqlite3.Connection:
        """Get a database connection."""
        conn = sqlite3.connect(self.DB_PATH)
        conn.execute("PRAGMA foreign_keys = ON")
        for key, value in DBConfig.PRAGMA.items():
            conn.execute(f"PRAGMA {key} = {value}")
        return conn

# Create a singleton instance of DBManager
db_manager = DBManager()

