"""
Enhanced UI theme with comprehensive styles for all widgets and states,
including support for advanced search dialog, video cards, player controls,
and all application components.
"""
from typing import ClassVar

from typing import ClassVar

class Theme:
    """Enhanced UI theme with support for all widgets and states"""
    # Primary colors
    PRIMARY = "#6a5acd"
    PRIMARY_HOVER = "#7a6bdb"
    PRIMARY_PRESSED = "#5a4bc9"
    PRIMARY_DISABLED = "#4a3bc9"
    SECONDARY = "#AAAAAA"
    TERTIARY = "#6b46c1"
    TERTIARY_HOVER = "#7b56d1"
    SUCCESS = "#00FF66"
    WARNING = "#FFCC00"
    DANGER = "#ff4444"
    INFO = "#FF9800"

    # Background colors
    BACKGROUND = "#0F0F0F"
    BACKGROUND_LIGHT = "#181818"
    SURFACE = "#212121"
    SURFACE_HOVER = "#2a2a2a"
    SURFACE_PRESSED = "#333333"
    SURFACE_LIGHT = "#282828"
    SURFACE_ACTIVE = "#333333"  # Added for active state

    # Text colors
    TEXT_PRIMARY = "#FFFFFF"
    TEXT_SECONDARY = "#AAAAAA"
    TEXT_DISABLED = "#6B6B6B"
    TEXT_LINK = "#6a5acd"

    # Borders
    BORDER = "#2A2A2A"
    BORDER_RADIUS: ClassVar[str] = "8px"
    BORDER_WIDTH: ClassVar[str] = "1px"

    # Spacing
    PADDING_SMALL: ClassVar[str] = "8px"
    PADDING_MEDIUM: ClassVar[str] = "12px"
    PADDING_LARGE: ClassVar[str] = "16px"
    MARGIN_SMALL: ClassVar[str] = "4px"
    MARGIN_MEDIUM: ClassVar[str] = "8px"
    MARGIN_LARGE: ClassVar[str] = "16px"

    # Font family
    FONT_FAMILY: ClassVar[str] = "'Inter', 'Segoe UI', 'Roboto', sans-serif"
    FONT_SIZE_SMALL: ClassVar[str] = "12px"
    FONT_SIZE_MEDIUM: ClassVar[str] = "14px"
    FONT_SIZE_LARGE: ClassVar[str] = "16px"

    @classmethod
    def global_stylesheet(cls) -> str:
        """Return the global stylesheet for the application"""
        return f"""
        /* ========== GLOBAL STYLES ========== */
        QMainWindow, QWidget {{
            background: {cls.BACKGROUND};
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            font-size: {cls.FONT_SIZE_MEDIUM};
        }}

        QToolTip {{
            background: {cls.SURFACE};
            color: {cls.TEXT_PRIMARY};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            padding: {cls.PADDING_SMALL};
            font-size: {cls.FONT_SIZE_SMALL};
        }}

        /* ========== BUTTONS ========== */
        /* Primary Button */
        QPushButton {{
            background: {cls.PRIMARY};
            color: white;
            border: none;
            border-radius: 6px;
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            font-family: {cls.FONT_FAMILY};
            min-height: 32px;
            font-size: {cls.FONT_SIZE_MEDIUM};
        }}
        QPushButton:hover {{
            background: {cls.PRIMARY_HOVER};
        }}
        QPushButton:pressed {{
            background: {cls.PRIMARY_PRESSED};
        }}
        QPushButton:disabled {{
            background: {cls.PRIMARY_DISABLED};
            color: {cls.TEXT_DISABLED};
        }}

        /* Secondary Button */
        QPushButton.secondary {{
            background: {cls.SURFACE};
            color: {cls.TEXT_PRIMARY};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            font-family: {cls.FONT_FAMILY};
            min-height: 32px;
        }}
        QPushButton.secondary:hover {{
            background: {cls.SURFACE_HOVER};
        }}
        QPushButton.secondary:pressed {{
            background: {cls.SURFACE_PRESSED};
        }}

        /* Danger Button */
        QPushButton.danger {{
            background: {cls.DANGER};
            color: white;
            border: none;
            border-radius: 6px;
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            font-family: {cls.FONT_FAMILY};
            min-height: 32px;
        }}
        QPushButton.danger:hover {{
            background: #ff5555;
        }}

        /* Tertiary Button */
        QPushButton.tertiary {{
            background: {cls.TERTIARY};
            color: white;
            border: none;
            border-radius: 6px;
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            font-family: {cls.FONT_FAMILY};
            min-height: 32px;
        }}
        QPushButton.tertiary:hover {{
            background: {cls.TERTIARY_HOVER};
        }}

        /* Media Control Buttons */
        QPushButton#mediaButton {{
            background: rgba(0, 0, 0, 0.3);
            color: white;
            border-radius: 24px;
            padding: {cls.PADDING_MEDIUM};
            min-width: 48px;
            min-height: 48px;
            font-size: {cls.FONT_SIZE_LARGE};
        }}
        QPushButton#mediaButton:hover {{
            background: rgba(0, 0, 0, 0.5);
        }}
        QPushButton#mediaButton:pressed {{
            background: rgba(0, 0, 0, 0.7);
        }}

        /* ========== INPUT FIELDS ========== */
        QLineEdit, QTextEdit, QPlainTextEdit, QComboBox {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            color: {cls.TEXT_PRIMARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            font-family: {cls.FONT_FAMILY};
            font-size: {cls.FONT_SIZE_MEDIUM};
            min-height: 32px;
        }}
        QLineEdit:focus, QTextEdit:focus, QPlainTextEdit:focus, QComboBox:focus {{
            border: {cls.BORDER_WIDTH} solid {cls.PRIMARY};
        }}

        /* ComboBox specific styles */
        QComboBox::drop-down {{
            border: 0px;
            width: 24px;
        }}
        QComboBox::down-arrow {{
            image: url(:/icons/down-arrow);
            width: 12px;
            height: 12px;
        }}
        QComboBox QAbstractItemView {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            color: {cls.TEXT_PRIMARY};
            selection-background-color: {cls.PRIMARY};
            outline: none;
        }}

        /* ========== SLIDERS ========== */
        QSlider::groove:horizontal {{
            background: {cls.BORDER};
            height: 4px;
            border-radius: 2px;
        }}
        QSlider::handle:horizontal {{
            background: {cls.PRIMARY};
            width: 12px;
            margin: -4px 0;
            border-radius: 6px;
        }}
        QSlider::handle:horizontal:hover {{
            background: {cls.PRIMARY_HOVER};
        }}

        /* ========== SCROLLBARS ========== */
        QScrollBar:vertical {{
            background: {cls.BACKGROUND_LIGHT};
            width: 12px;
            border-radius: 6px;
            margin: 0px;
        }}
        QScrollBar::handle:vertical {{
            background: {cls.PRIMARY};
            border-radius: 6px;
            min-height: 20px;
        }}
        QScrollBar::handle:vertical:hover {{
            background: {cls.PRIMARY_HOVER};
        }}
        QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {{
            background: none;
            border: none;
            height: 0px;
        }}
        QScrollBar::sub-page:vertical, QScrollBar::add-page:vertical {{
            background: none;
        }}


        /* ========== PROGRESS BARS ========== */
        QProgressBar {{
            background: {cls.BORDER};
            border-radius: 6px;
            text-align: center;
            border: none;
            min-height: 8px;
        }}
        QProgressBar::chunk {{
            background: {cls.PRIMARY};
            border-radius: 6px;
        }}

        /* ========== CHECKBOXES ========== */
        QCheckBox::indicator {{
            width: 16px;
            height: 16px;
            border-radius: 4px;
        }}
        QCheckBox::indicator:checked {{
            background: {cls.PRIMARY};
            border: none;
            border-radius: 4px;
        }}
        QCheckBox::indicator:unchecked {{
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 4px;
        }}

        /* ========== RADIO BUTTONS ========== */
        QRadioButton::indicator {{
            width: 16px;
            height: 16px;
            border-radius: 8px;
        }}
        QRadioButton::indicator:checked {{
            background: {cls.PRIMARY};
            border: none;
            border-radius: 8px;
        }}

        /* ========== MENUS ========== */
        QMenu {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            padding: {cls.PADDING_SMALL};
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
        }}
        QMenu::item {{
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            min-height: 32px;
        }}
        QMenu::item:selected {{
            background: rgba(106, 90, 205, 0.2);
            border-radius: 4px;
        }}
        QMenu::item:disabled {{
            color: {cls.TEXT_DISABLED};
        }}

        /* ========== LIST WIDGETS ========== */
        QListWidget {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            padding: 0px;
            outline: none;
        }}
        QListWidget::item {{
            background: {cls.SURFACE};
            border-bottom: {cls.BORDER_WIDTH} solid {cls.BORDER};
            padding: {cls.PADDING_MEDIUM} {cls.PADDING_SMALL};
            font-family: {cls.FONT_FAMILY};
            min-height: 48px;
        }}
        QListWidget::item:selected {{
            background: {cls.PRIMARY};
            color: white;
            border-radius: 4px;
        }}
        QListWidget::item:hover {{
            background: {cls.SURFACE_HOVER};
        }}

        /* ========== TAB WIDGETS ========== */
        QTabWidget::pane {{
            border: none;
            background: transparent;
        }}
        QTabWidget::tab-bar {{
            alignment: left;
        }}
        QTabBar::tab {{
            background: {cls.BACKGROUND_LIGHT};
            color: {cls.TEXT_SECONDARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            border: none;
            border-bottom: {cls.BORDER_WIDTH} solid {cls.BORDER};
            min-width: 120px;
            min-height: 36px;
            font-family: {cls.FONT_FAMILY};
        }}
        QTabBar::tab:selected {{
            background: {cls.SURFACE};
            color: {cls.TEXT_PRIMARY};
            border-bottom: {cls.BORDER_WIDTH} solid {cls.PRIMARY};
        }}
        QTabBar::tab:hover {{
            background: {cls.SURFACE_HOVER};
        }}

        /* ========== GROUP BOXES ========== */
        QGroupBox {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            margin-top: 16px;
            padding-top: 20px;
        }}
        QGroupBox::title {{
            subcontrol-origin: margin;
            left: {cls.PADDING_MEDIUM};
            padding: 0 5px;
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            font-size: {cls.FONT_SIZE_MEDIUM};
            font-weight: bold;
        }}

        /* ========== FRAMES ========== */
        QFrame {{
            background: transparent;
            border: none;
        }}
        QFrame[role="card"] {{
            background: {cls.SURFACE};
            border-radius: {cls.BORDER_RADIUS};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
        }}
        QFrame[role="card"]:hover {{
            border: {cls.BORDER_WIDTH} solid {cls.PRIMARY};
        }}

        /* ========== LABELS ========== */
        QLabel {{
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            font-size: {cls.FONT_SIZE_MEDIUM};
        }}
        QLabel[role="title"] {{
            font-weight: bold;
            font-size: {cls.FONT_SIZE_LARGE};
        }}
        QLabel[role="subtitle"] {{
            color: {cls.TEXT_SECONDARY};
            font-size: {cls.FONT_SIZE_SMALL};
        }}
        QLabel[role="caption"] {{
            color: {cls.TEXT_SECONDARY};
            font-size: 11px;
        }}

        /* ========== STATUS BAR ========== */
        QStatusBar {{
            background: {cls.SURFACE};
            color: {cls.TEXT_SECONDARY};
            font-family: {cls.FONT_FAMILY};
            padding: {cls.PADDING_SMALL};
            border-top: {cls.BORDER_WIDTH} solid {cls.BORDER};
        }}

        /* ========== DIALOGS ========== */
        QDialog {{
            background: {cls.BACKGROUND};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 8px;
        }}
        QDialog QLabel {{
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
        }}
        QDialog QPushButton {{
            min-width: 80px;
        }}

        /* ========== ADVANCED SEARCH DIALOG ========== */
        QDialog#AdvancedSearchDialog {{
            background: {cls.BACKGROUND};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 8px;
            min-width: 500px;
        }}
        QDialog#AdvancedSearchDialog QLabel {{
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            font-size: {cls.FONT_SIZE_MEDIUM};
            margin-bottom: {cls.MARGIN_SMALL};
        }}
        QDialog#AdvancedSearchDialog QLineEdit {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            color: {cls.TEXT_PRIMARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            min-height: 36px;
            font-size: {cls.FONT_SIZE_MEDIUM};
        }}
        QDialog#AdvancedSearchDialog QComboBox {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            color: {cls.TEXT_PRIMARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            min-height: 36px;
            font-size: {cls.FONT_SIZE_MEDIUM};
        }}
        QDialog#AdvancedSearchDialog QPushButton {{
            min-height: 36px;
            padding: 0 {cls.PADDING_MEDIUM};
        }}
        QDialog#AdvancedSearchDialog QPushButton[role="primary"] {{
            background: {cls.PRIMARY};
            color: white;
        }}
        QDialog#AdvancedSearchDialog QPushButton[role="secondary"] {{
            background: {cls.SURFACE};
            color: {cls.TEXT_PRIMARY};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
        }}
        QDialog#AdvancedSearchDialog QGridLayout {{
            margin: {cls.MARGIN_MEDIUM};
            spacing: {cls.MARGIN_LARGE};
        }}

        /* ========== VIDEO CARD ========== */
        QFrame#videoCard {{
            background: {cls.SURFACE};
            border-radius: {cls.BORDER_RADIUS};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            padding: 0px;
        }}
        QFrame#videoCard:hover {{
            border: {cls.BORDER_WIDTH} solid {cls.PRIMARY};
        }}
        QLabel#thumb_label {{
            background: {cls.BACKGROUND};
            border-top-left-radius: 12px;
            border-top-right-radius: 12px;
            border: none;
        }}
        QLabel#title {{
            color: {cls.TEXT_PRIMARY};
            font-size: {cls.FONT_SIZE_MEDIUM};
            font-weight: 500;
            margin: {cls.MARGIN_SMALL} 0;
        }}
        QLabel#channel {{
            color: {cls.TEXT_SECONDARY};
            font-size: {cls.FONT_SIZE_SMALL};
        }}
        QLabel#duration {{
            background: rgba(0, 0, 0, 0.8);
            color: white;
            border-radius: 4px;
            padding: 2px 6px;
            font-size: 11px;
            font-weight: 500;
            min-width: 40px;
            text-align: center;
        }}
        QPushButton#playOverlay {{
            background: rgba(255, 255, 255, 0.9);
            color: black;
            font-size: 20px;
            border-radius: 4px;
            border: none;
            padding: 0;
            min-width: 48px;
            min-height: 36px;
        }}
        QPushButton#playOverlay:hover {{
            background: white;
        }}

        /* ========== PLAYER CONTROLS ========== */
        QFrame#playerControls {{
            background: rgba(0, 0, 0, 0.7);
            border-radius: 0;
            padding: {cls.PADDING_MEDIUM};
            margin: 0;
        }}
        QSlider#seekSlider::groove:horizontal {{
            background: {cls.BORDER};
            height: 4px;
            border-radius: 2px;
        }}
        QSlider#seekSlider::handle:horizontal {{
            background: {cls.PRIMARY};
            width: 12px;
            margin: -4px 0;
            border-radius: 6px;
        }}
        QSlider#volumeSlider::groove:vertical {{
            background: #121212;
            width: 8px;
            border-radius: 4px;
        }}
        QSlider#volumeSlider::handle:vertical {{
            background: {cls.PRIMARY};
            border: 1px solid #5c5c5c;
            width: 16px;
            height: 16px;
            margin: -5px -5px;
            border-radius: 8px;
        }}

        /* ========== SIDEBAR ========== */
        QFrame#sidebar {{
            background: {cls.BACKGROUND_LIGHT};
            border: none;
            border-right: {cls.BORDER_WIDTH} solid {cls.BORDER};
            min-width: 240px;
            max-width: 240px;
        }}
        QPushButton#sidebarButton {{
            background: transparent;
            color: {cls.TEXT_SECONDARY};
            border: none;
            border-radius: 6px;
            padding: {cls.PADDING_MEDIUM} {cls.PADDING_LARGE};
            text-align: left;
            font-size: {cls.FONT_SIZE_MEDIUM};
            font-family: {cls.FONT_FAMILY};
            min-height: 40px;
        }}
        QPushButton#sidebarButton:hover {{
            background: {cls.SURFACE_HOVER};
            color: {cls.TEXT_PRIMARY};
        }}
        QPushButton#sidebarButton:checked {{
            background: {cls.PRIMARY};
            color: white;
        }}

        /* ========== SEARCH BAR ========== */
        QLineEdit#searchBar {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 20px;
            color: {cls.TEXT_PRIMARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_LARGE};
            font-size: {cls.FONT_SIZE_MEDIUM};
            min-height: 36px;
            font-family: {cls.FONT_FAMILY};
            margin-bottom: 20px;
        }}
        QLineEdit#searchBar:focus {{
            border: {cls.BORDER_WIDTH} solid {cls.PRIMARY};
        }}

        /* ========== HERO SECTION ========== */
        QFrame#heroWidget {{
            background: qlineargradient(x1:0, y1:0, x2:1, y2:0,
                stop:0 #6A1B9A, stop:0.5 #8E24AA, stop:1 #4A148C);
            margin-top: 0px;
            padding: {cls.PADDING_LARGE};
        }}
        """

    @classmethod
    def search_bar_style(cls) -> str:
        """Return stylesheet for search bar"""
        return f"""
        QLineEdit#searchBar {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 20px;
            color: {cls.TEXT_PRIMARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_LARGE};
            font-size: {cls.FONT_SIZE_MEDIUM};
            min-height: 36px;
            font-family: {cls.FONT_FAMILY};
        }}
        QLineEdit#searchBar:focus {{
            border: {cls.BORDER_WIDTH} solid {cls.PRIMARY};
        }}
        """

    @classmethod
    def sidebar_style(cls) -> str:
        """Return stylesheet for sidebar"""
        return f"""
        QFrame#sidebar {{
            background: {cls.BACKGROUND_LIGHT};
            border: none;
            border-right: {cls.BORDER_WIDTH} solid {cls.BORDER};
            min-width: 240px;
            max-width: 240px;
        }}
        QPushButton#sidebarButton {{
            background: transparent;
            color: {cls.TEXT_SECONDARY};
            border: none;
            border-radius: 6px;
            padding: {cls.PADDING_MEDIUM} {cls.PADDING_LARGE};
            text-align: left;
            font-size: {cls.FONT_SIZE_MEDIUM};
            font-family: {cls.FONT_FAMILY};
            min-height: 40px;
        }}
        QPushButton#sidebarButton:hover {{
            background: {cls.SURFACE_HOVER};
            color: {cls.TEXT_PRIMARY};
        }}
        QPushButton#sidebarButton:checked {{
            background: {cls.PRIMARY};
            color: white;
        }}
        """

    @classmethod
    def video_card_style(cls) -> str:
        """Return stylesheet for video cards"""
        return f"""
        QFrame#videoCard {{
            background: {cls.SURFACE};
            border-radius: {cls.BORDER_RADIUS};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            padding: 0px;
        }}
        QFrame#videoCard:hover {{
            border: {cls.BORDER_WIDTH} solid {cls.PRIMARY};
        }}
        QLabel#thumb_label {{
            background: {cls.BACKGROUND};
            border-top-left-radius: 12px;
            border-top-right-radius: 12px;
            border: none;
        }}
        QLabel#title {{
            color: {cls.TEXT_PRIMARY};
            font-size: {cls.FONT_SIZE_MEDIUM};
            font-weight: 500;
            margin: {cls.MARGIN_SMALL} 0;
        }}
        QLabel#channel {{
            color: {cls.TEXT_SECONDARY};
            font-size: {cls.FONT_SIZE_SMALL};
        }}
        QLabel#duration {{
            background: rgba(0, 0, 0, 0.8);
            color: white;
            border-radius: 4px;
            padding: 2px 6px;
            font-size: 11px;
            font-weight: 500;
            min-width: 40px;
            text-align: center;
        }}
        QPushButton#playOverlay {{
            background: rgba(255, 255, 255, 0.9);
            color: black;
            font-size: 20px;
            border-radius: 4px;
            border: none;
            padding: 0;
            min-width: 48px;
            min-height: 36px;
        }}
        QPushButton#playOverlay:hover {{
            background: white;
        }}
        """

    @classmethod
    def volume_slider_style(cls) -> str:
        """Return stylesheet for volume slider"""
        return f"""
        QSlider#volumeSlider {{
            background: transparent;
            margin: 0px;
        }}
        QSlider::groove:vertical {{
            background: #121212;
            width: 8px;
            border-radius: 4px;
        }}
        QSlider::handle:vertical {{
            background: {cls.PRIMARY};
            border: 1px solid #5c5c5c;
            width: 16px;
            height: 16px;
            margin: -5px -5px;
            border-radius: 8px;
        }}
        QSlider::add-page:vertical {{
            background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                stop:0 {cls.PRIMARY}, stop:1 #121212);
            border-radius: 4px;
        }}
        QSlider::sub-page:vertical {{
            background: #333;
            border-radius: 4px;
        }}
        """

    @classmethod
    def player_controls_style(cls) -> str:
        """Return stylesheet for player controls"""
        return f"""
        QFrame#playerControls {{
            background: rgba(0, 0, 0, 0.7);
            border-radius: 0;
            padding: {cls.PADDING_MEDIUM};
            margin: 0;
        }}
        QSlider#seekSlider::groove:horizontal {{
            background: {cls.BORDER};
            height: 4px;
            border-radius: 2px;
        }}
        QSlider#seekSlider::handle:horizontal {{
            background: {cls.PRIMARY};
            width: 12px;
            margin: -4px 0;
            border-radius: 6px;
        }}
        """ + cls.volume_slider_style()

    @classmethod
    def list_widget_style(cls) -> str:
        """Return stylesheet for list widgets"""
        return f"""
        QListWidget {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            padding: 0px;
            outline: none;
        }}
        QListWidget::item {{
            background: {cls.SURFACE};
            border-bottom: {cls.BORDER_WIDTH} solid {cls.BORDER};
            padding: {cls.PADDING_MEDIUM} {cls.PADDING_SMALL};
            font-family: {cls.FONT_FAMILY};
            min-height: 48px;
        }}
        QListWidget::item:selected {{
            background: {cls.PRIMARY};
            color: white;
            border-radius: 4px;
        }}
        QListWidget::item:hover {{
            background: {cls.SURFACE_HOVER};
        }}
        """

    @classmethod
    def advanced_search_style(cls) -> str:
        """Return stylesheet for advanced search dialog"""
        return f"""
        QDialog#AdvancedSearchDialog {{
            background: {cls.BACKGROUND};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 8px;
            min-width: 500px;
        }}
        QDialog#AdvancedSearchDialog QLabel {{
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            font-size: {cls.FONT_SIZE_MEDIUM};
            margin-bottom: {cls.MARGIN_SMALL};
        }}
        QDialog#AdvancedSearchDialog QLineEdit {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            color: {cls.TEXT_PRIMARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            min-height: 36px;
            font-size: {cls.FONT_SIZE_MEDIUM};
        }}
        QDialog#AdvancedSearchDialog QComboBox {{
            background: {cls.SURFACE};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
            border-radius: 6px;
            color: {cls.TEXT_PRIMARY};
            padding: {cls.PADDING_SMALL} {cls.PADDING_MEDIUM};
            min-height: 36px;
            font-size: {cls.FONT_SIZE_MEDIUM};
        }}
        QDialog#AdvancedSearchDialog QPushButton {{
            min-height: 36px;
            padding: 0 {cls.PADDING_MEDIUM};
        }}
        QDialog#AdvancedSearchDialog QPushButton[role="primary"] {{
            background: {cls.PRIMARY};
            color: white;
        }}
        QDialog#AdvancedSearchDialog QPushButton[role="secondary"] {{
            background: {cls.SURFACE};
            color: {cls.TEXT_PRIMARY};
            border: {cls.BORDER_WIDTH} solid {cls.BORDER};
        }}
        QDialog#AdvancedSearchDialog QGridLayout {{
            margin: {cls.MARGIN_MEDIUM};
            spacing: {cls.MARGIN_LARGE};
        }}
        """

    @classmethod
    def hero_section_style(cls) -> str:
        """Return stylesheet for hero section"""
        return f"""
        QFrame#heroWidget {{
            background: qlineargradient(x1:0, y1:0, x2:1, y2:0,
                stop:0 #6A1B9A, stop:0.5 #8E24AA, stop:1 #4A148C);
            margin-top: 0px;
            padding: {cls.PADDING_LARGE};
        }}
        QFrame#heroWidget QLabel {{
            color: white;
        }}
        QFrame#heroWidget QLabel[role="title"] {{
            font-size: {cls.FONT_SIZE_LARGE};
            font-weight: bold;
        }}
        QFrame#heroWidget QLabel[role="subtitle"] {{
            font-size: {cls.FONT_SIZE_SMALL};
            opacity: 0.8;
        }}
        """

