"""
Logging configuration for PurpleTube Player.
This module sets up the logging infrastructure for the application,
including file rotation, log levels, and error handling.
"""
import logging
import sys
from pathlib import Path
from logging.handlers import RotatingFileHandler
from .settings import DATA_DIR, LOG_PATH, LogConfig

def setup_logging() -> logging.Logger:
    """
    Setup logging configuration for the application.
    Configures both file-based logging (with rotation) and console logging.
    Falls back to basic console logging if file setup fails.
    Returns:
        logging.Logger: Configured logger instance.
    """
    logger = logging.getLogger("PurpleTube")
    logger.setLevel(LogConfig.LOG_LEVEL)

    # Create formatter
    formatter = logging.Formatter(
        fmt=LogConfig.LOG_FORMAT,
        datefmt='%Y-%m-%d %H:%M:%S'
    )

    # Ensure log directory exists
    try:
        DATA_DIR.mkdir(parents=True, exist_ok=True)
        if sys.platform != "win32":
            DATA_DIR.chmod(0o700)  # Only owner can read/write/execute
    except Exception as e:
        print(f"Failed to create log directory: {e}", file=sys.stderr)

    # File handler with rotation
    try:
        file_handler = RotatingFileHandler(
            filename=LOG_PATH,
            maxBytes=LogConfig.MAX_LOG_SIZE,
            backupCount=LogConfig.LOG_BACKUP_COUNT,
            encoding=LogConfig.LOG_ENCODING,
            delay=True
        )
        file_handler.setFormatter(formatter)
        logger.addHandler(file_handler)
    except Exception as e:
        print(f"Failed to setup file logging: {e}", file=sys.stderr)

    # Console handler (always add this)
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setFormatter(formatter)
    logger.addHandler(console_handler)

    return logger

# Initialize logger
logger = setup_logging()

# Log setup completion after logger is fully configured
logger.info("Logging setup complete")
logger.debug(f"Log file: {LOG_PATH}")
logger.debug(f"Log level: {logging.getLevelName(LogConfig.LOG_LEVEL)}")

